import { fetcher } from '@/utils/axios';

// Types for candidate forms
export interface CandidateFormData {
  candidateId: string;
  formId: string;
  accessToken: string;
  responses: Record<string, any>;
}

// Correspond à votre FieldDTO.java + champs supplémentaires pour le frontend
export interface FormField {
  id: number; // Long en Java = number en TypeScript
  type: string; // FieldType as string
  label: string;
  placeholder?: string;
  isRequired: boolean; // isRequired en Java
  options?: FormFieldOption[];
  // Champs supplémentaires pour le frontend (non présents dans FieldDTO.java)
  name: string; // Nom du champ pour identifier les réponses
}

// Correspond à votre FieldOptionDTO.java
export interface FormFieldOption {
  id: number; // Long en Java
  value: string;
  label: string;
}

// Correspond à votre FormsDTO.java
export interface CandidateForm {
  id: number; // Long en Java
  title: string;
  description: string;
  creator?: string;
  category?: string;
  department?: string;
  createdDate?: string;
  lastModifiedDate?: string;
  fields: FormField[];
  // Ajout des informations spécifiques au candidat
  candidate: {
    id: string;
    firstName: string;
    lastName: string;
    email: string;
  };
  expiryDate: string;
  isExpired: boolean;
  isCompleted: boolean;
}

export interface FormSubmissionRequest {
  accessToken: string;
  responses: Record<string, any>;
}

export interface FormSubmissionResponse {
  success: boolean;
  message: string;
}

// API endpoints for candidate forms
export const candidateFormApi = {
  // Get form by access token
  getFormByToken: async (token: string, signal?: AbortSignal): Promise<CandidateForm> => {
    return fetcher([`/api/hr/public/candidate-form/${token}`, { signal }], 'get');
  },

  // Submit form responses
  submitForm: async (data: FormSubmissionRequest): Promise<FormSubmissionResponse> => {
    return fetcher(['/api/hr/public/candidate-form/submit', {}], 'post', data);
  },

  // Upload file for form
  uploadFormFile: async (token: string, fieldName: string, file: File): Promise<{ fileUrl: string }> => {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('fieldName', fieldName);
    
    return fetcher([
      `/api/hr/public/candidate-form/${token}/upload`,
      { headers: { 'Content-Type': 'multipart/form-data' } }
    ], 'post', formData);
  },
};
