const endpointPrefix = '/api/hr/forms';

export const formsEndpoints = {
  base: endpointPrefix,
  create: `${endpointPrefix}/create`,
  byId: (id: string) => `${endpointPrefix}/${id}`,
  update: (id: string) => `${endpointPrefix}/${id}/update`,
  delete: (id: string) => `${endpointPrefix}/${id}/delete`,
  all: `${endpointPrefix}/all`,
  search: `${endpointPrefix}/search`,
};

export interface FormDTO {
  id: number | string;
  title: string;
  description: string;
  creator?: string;
  category?: string;
  department?: string;
  createdDate?: string;
  lastModifiedDate?: string;
  fields?: any[];
}

export interface CreateFormRequest {
  title: string;
  description: string;
  creator?: string;
  category?: string;
  department?: string;
}

export interface PageResponse<T> {
  content: T[];
  totalElements: number;
  totalPages: number;
  size: number;
  number: number;
  first: boolean;
  last: boolean;
}

import { fetcher } from '@/utils/axios';

export const formsApi = {
  createForm: async (data: CreateFormRequest): Promise<{ message: string; id: string }> => {
    return fetcher([formsEndpoints.create, {}], 'post', data);
  },

  getFormById: async (id: string): Promise<FormDTO> => {
    return fetcher(formsEndpoints.byId(id), 'get');
  },

  getAllForms: async (page: number = 0, size: number = 20): Promise<PageResponse<FormDTO>> => {
    return fetcher([formsEndpoints.all, { params: { page, size } }], 'get');
  },

  updateForm: async (id: string, data: Partial<FormDTO>): Promise<{ message: string }> => {
    return fetcher([formsEndpoints.update(id), {}], 'put', data);
  },

  deleteForm: async (id: string): Promise<{ message: string }> => {
    return fetcher([formsEndpoints.delete(id), {}], 'delete');
  },

  searchForms: async (query: string, page: number = 0, size: number = 20): Promise<PageResponse<FormDTO>> => {
    return fetcher([formsEndpoints.search, { params: { q: query, page, size } }], 'get');
  }
};
