
import type { 
  IWorkingHours, 
  CreateWorkingHoursRequest, 
  UpdateWorkingHoursRequest 
} from '@/shared/types/working-hours';
import axiosInstance from '@/utils/axios';

const workingHoursEndpoints = {
  base: '/api/hr/working-hours',
  byEmployee: (employeeId: string) => `/api/hr/working-hours/employee/${employeeId}`,
  byDay: (day: number) => `/api/hr/working-hours/day/${day}`,
  available: '/api/hr/working-hours/available',
  byId: (id: string) => `/api/hr/working-hours/${id}`,
};

export const workingHoursApi = {

  getWorkingHoursByEmployeeId: async (employeeId: string): Promise<{ data: IWorkingHours[] }> => {
    const response = await axiosInstance.get(workingHoursEndpoints.byEmployee(employeeId));
    return response;
  },
  getWorkingHoursByDay: async (day: number): Promise<{ data: IWorkingHours[] }> => {
    const response = await axiosInstance.get(workingHoursEndpoints.byDay(day));
    return response;
  },

  getAvailableEmployees: async (
    day: number, 
    startHour: number, 
    endHour: number
  ): Promise<{ data: string[] }> => {
    const response = await axiosInstance.get(workingHoursEndpoints.available, {
      params: { day, startHour, endHour }
    });
    return response;
  },
  createWorkingHours: async (request: CreateWorkingHoursRequest): Promise<{ data: IWorkingHours }> => {
    const response = await axiosInstance.post(workingHoursEndpoints.base, request);
    return response;
  },
  updateWorkingHours: async (
    id: string, 
    request: UpdateWorkingHoursRequest
  ): Promise<{ data: IWorkingHours }> => {
    const response = await axiosInstance.put(workingHoursEndpoints.byId(id), request);
    return response;
  },

  deleteWorkingHours: async (id: string): Promise<void> => {
    await axiosInstance.delete(workingHoursEndpoints.byId(id));
  },


  deleteAllWorkingHoursByEmployeeId: async (employeeId: string): Promise<void> => {
    await axiosInstance.delete(workingHoursEndpoints.byEmployee(employeeId));
  },
};

