// useDocumentRequestStore.ts

import { create } from "zustand";
import { AddDocumentRequestDTO, IDocumentRequest, UpdateRequestDto } from "../../../types/document-request";
import { documentRequestEndpoints } from "../../endpoints/admin-service/document-request";
import axiosInstance from "@/utils/axios";

interface DocumentRequestState {
  requests: IDocumentRequest[];
  userRequests: IDocumentRequest[];
  loading: boolean;
  error: string | null;

  fetchAll: () => Promise<void>;
  fetchAllForUser: (userUuid : string) => Promise<void>;
  fetchById: (id: number) => Promise<IDocumentRequest | null>;
  addRequest: (data: AddDocumentRequestDTO) => Promise<IDocumentRequest | null>;
  updateRequest: (data: UpdateRequestDto, codeReq : string) => Promise<IDocumentRequest | null>;
  deleteRequest: (id: number) => Promise<void>;
  activateRequest: (codeReq: string) => Promise<void>;
  deactivateRequest: (codeReq: string) => Promise<void>;
  approveRequest: (codeReq: string) => Promise<void>;
  rejectRequest: (codeReq: string) => Promise<void>;
}

export const useDocumentRequestStore = create<DocumentRequestState>((set) => ({
  requests: [],
  userRequests : [],
  loading: false,
  error: null,

  fetchAll: async () => {
    set({ loading: true, error: null });
    try {
      const res = await axiosInstance.get<IDocumentRequest[]>(documentRequestEndpoints.getAll);
      set({ requests: res.data });
    } catch (error: any) {
      set({ error: error.message });
    } finally {
      set({ loading: false });
    }
  },
  
  fetchAllForUser: async (userUuid) => {
    set({ loading: true, error: null });
    try {
      const res = await axiosInstance.get<IDocumentRequest[]>(documentRequestEndpoints.getAllForUser(userUuid));
      set({ userRequests: res.data });
    } catch (error: any) {
      set({ error: error.message });
    } finally {
      set({ loading: false });
    }
  },

  fetchById: async (id) => {
    try {
      const res = await axiosInstance.get<IDocumentRequest>(documentRequestEndpoints.getById(id));
      return res.data;
    } catch {
      return null;
    }
  },

  addRequest: async (data) : Promise<IDocumentRequest | null> => {
    try {
      const response = await axiosInstance.post(documentRequestEndpoints.add, data);
      return response.data;
    } catch (error: any) {
      set({ error: error.message });
      return null;
    }
  },

  updateRequest: async (data, codeReq) : Promise<IDocumentRequest | null>  => {
    try {
      const res = await axiosInstance.put(documentRequestEndpoints.update(codeReq), data);
      return res.data;
    } catch (error: any) {
      set({ error: error.message });
      return null;
    }
  },

  deleteRequest: async (id) => {
    try {
      await axiosInstance.delete(documentRequestEndpoints.delete(id));
    } catch (error: any) {
      set({ error: error.message });
    }
  },
  activateRequest: async (codeReq) => {
    try {
      await axiosInstance.put(documentRequestEndpoints.activate(codeReq));
    } catch (error: any) {
      set({ error: error.message });
    }
  },

  deactivateRequest: async (codeReq) => {
    try {
      await axiosInstance.put(documentRequestEndpoints.deactivate(codeReq));
    } catch (error: any) {
      set({ error: error.message });
    }
  },

  approveRequest: async (codeReq) => {
    try {
      await axiosInstance.put(documentRequestEndpoints.approve(codeReq));
    } catch (error: any) {
      set({ error: error.message });
    }
  },

  rejectRequest: async (codeReq) => {
    try {
      await axiosInstance.put(documentRequestEndpoints.reject(codeReq));
    } catch (error: any) {
      set({ error: error.message });
    }
  },
}));
