// stores/document-store.ts
import { create } from 'zustand';
import { documentEndpoints } from '../../endpoints/document-service';
import axiosInstance from '@/utils/axios';
import type { DocumentDTO, AddDocumentRequest, BulkDocumentRequest, BulkMoveRequest } from '@/shared/types/document';
import { da } from 'date-fns/locale';

interface DocumentStoreState {
  documents: DocumentDTO[];
  favorites: DocumentDTO[];
  error?: string;

  getAll: (ownerUuid: string | null) => Promise<DocumentDTO[]>;
  getDetails: (documentId: number, ownerUuid: string) => Promise<DocumentDTO>;
  getRoot: (ownerUuid: string) => Promise<DocumentDTO[]>;
  getFavorites: (ownerUuid: string) => Promise<void>;
  download: (documentId: number, ownerUuid: string) => Promise<Blob>;
  getDownloadUrl: (documentId: number, ownerUuid: string) => Promise<string>;

  search: (ownerUuid: string, keyword?: string, sortBy?: string, asc?: boolean) => Promise<DocumentDTO[]>;
  advancedSearch: (params: Record<string, any>) => Promise<void>;

  upload: (file: File, data: AddDocumentRequest) => Promise<DocumentDTO>;
  delete: (documentId: number, ownerUuid: string) => Promise<void>;
  bulkDelete: (req: BulkDocumentRequest) => Promise<void>;

  share: (documentId: number, targetUserUuid: string) => Promise<void>;
  bulkShare: (req: BulkDocumentRequest) => Promise<void>;

  markFavorite: (documentId: number, ownerUuid: string) => Promise<void>;
  unmarkFavorite: (documentId: number, ownerUuid: string) => Promise<void>;

  move: (documentId: number, ownerUuid: string, folderId?: number) => Promise<DocumentDTO>;
  bulkMove: (req: BulkMoveRequest) => Promise<void>;

  rename: (documentId: number, newName: string, ownerUuid: string) => Promise<void>;
}

export const useDocumentStore = create<DocumentStoreState>((set) => ({
  documents: [],
  favorites: [],

  async getAll(ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.getAll(ownerUuid));
    set({ documents: data });
    return data;
  },

  async getDetails(documentId, ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.getDetails(documentId, ownerUuid));
    return data;
  },

  async getRoot(ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.getRoot(ownerUuid));
    set({ documents: data });
    return data;
  },

  async getFavorites(ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.favorites(ownerUuid));
    set({ favorites: data });
  },

  async download(documentId, ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.download(documentId, ownerUuid), {
      responseType: 'blob'
    });
    return data;
  },

  async getDownloadUrl(documentId, ownerUuid) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.getDownloadUrl(documentId, ownerUuid));
    return data;
  },

  async search(ownerUuid, keyword = '', sortBy = 'name', asc = true) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.search(ownerUuid, keyword, sortBy, asc));
    set({ documents: data });
    return data;
  },

  async advancedSearch(params) {
    const { data } = await axiosInstance.get(documentEndpoints.UserDocument.advancedSearch, { params });
    set({ documents: data.content });
  },

  async upload(file, requestData) {
    const formData = new FormData();
    formData.append('file', file);
    Object.entries(requestData).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        formData.append(key, value.toString());
      }
    });
    const { data } = await axiosInstance.post(documentEndpoints.UserDocument.upload, formData, {
      headers: { 'Content-Type': 'multipart/form-data' }
    });
    return data;
  },

  async delete(documentId, ownerUuid) {
    await axiosInstance.delete(documentEndpoints.UserDocument.delete(documentId, ownerUuid));
  },

  async bulkDelete(req) {
    await axiosInstance.post(documentEndpoints.UserDocument.bulkDelete, req);
  },

  async share(documentId, targetUserUuid) {
    await axiosInstance.post(documentEndpoints.UserDocument.share(documentId, targetUserUuid));
  },

  async bulkShare(req) {
    await axiosInstance.post(documentEndpoints.UserDocument.bulkShare, req);
  },

  async markFavorite(documentId, ownerUuid) {
    await axiosInstance.post(documentEndpoints.UserDocument.markFavorite(documentId, ownerUuid));
  },

  async unmarkFavorite(documentId, ownerUuid) {
    await axiosInstance.delete(documentEndpoints.UserDocument.unmarkFavorite(documentId, ownerUuid));
  },

  async move(documentId, ownerUuid, folderId)  {
    const response = await axiosInstance.post(documentEndpoints.UserDocument.move(documentId, ownerUuid, folderId));
    return response.data;
  },

  async bulkMove(req) {
    await axiosInstance.post(documentEndpoints.UserDocument.bulkMove, req);
  },

  async rename(documentId, newName, ownerUuid) {
    await axiosInstance.post(documentEndpoints.UserDocument.rename(documentId, newName, ownerUuid));
  }
}));
