import {create} from 'zustand';
import axiosInstance from '@/utils/axios';
import { FuelSupplier } from '@/shared/types/fleet-management/fuelSupplier';
import { FuelSupplierEndpoints } from '../../endpoints/fleet-service/fuel-supplier-endpoints';

interface FuelSupplierStore {
  fuelSuppliers: FuelSupplier[];
  loading: boolean;
  error: string | null;
  
  fetchAllFuelSuppliers: () => Promise<void>;
  fetchFuelSupplierById: (id: number) => Promise<FuelSupplier | null>;
  createFuelSupplier: (data: any) => Promise<FuelSupplier>;
  updateFuelSupplier: (id: number, data: any) => Promise<FuelSupplier>;
  deleteFuelSupplier: (id: number) => Promise<void>;
  toggleFuelSupplierActive: (id: number) => Promise<void>;
  startCsvExport: (params : any, userId : number) => Promise<void>;
  downloadCsvExport: (filePath: string) => Promise<void>;
}

export const useFuelSupplierStore = create<FuelSupplierStore>((set) => ({
  fuelSuppliers: [],
  loading: false,
  error: null,
  
  fetchAllFuelSuppliers: async () => {
    set({ loading: true, error : null });
    try {
      const response = await axiosInstance.get(FuelSupplierEndpoints.GET_ALL);
      set({ fuelSuppliers: response.data.content });
    } catch (error) {
      set({ error: 'Failed to fetch fuel suppliers' });
    } finally {
      set({ loading: false });
    }
  },
  
  fetchFuelSupplierById: async (id: number) => {
    set({ loading: true });
    try {
      const response = await axiosInstance.get(FuelSupplierEndpoints.GET_BY_ID(id));
      return response.data;
    } catch (error) {
      set({ error: 'Failed to fetch fuel supplier' });
      return null;
    } finally {
      set({ loading: false });
    }
  },
  
  createFuelSupplier: async (data: any) => {
    try {
      const response = await axiosInstance.post(FuelSupplierEndpoints.CREATE, data);
      return response.data;
    } catch (error) {
      set({ error: 'Failed to create fuel supplier' });
      throw error;
    }
  },
  
  updateFuelSupplier: async (id: number, data: any) => {
    try {
      const response = await axiosInstance.put(FuelSupplierEndpoints.UPDATE(id), data);
      return response.data;
    } catch (error) {
      set({ error: 'Failed to update fuel supplier' });
      throw error;
    }
  },
  
  deleteFuelSupplier: async (id: number) => {
    try {
      await axiosInstance.delete(FuelSupplierEndpoints.DELETE(id));
    } catch (error) {
      set({ error: 'Failed to delete fuel supplier' });
      throw error;
    }
  },
  
  toggleFuelSupplierActive: async (id: number) => {
    try {
      await axiosInstance.post(FuelSupplierEndpoints.TOGGLE_ACTIVE(id));
    } catch (error) {
      set({ error: 'Failed to toggle active status' });
      throw error;
    }
  },
  startCsvExport: async (params, userId) => {
        set({ loading: true, error: null });
        try {
            await axiosInstance.post(FuelSupplierEndpoints.START_EXPORT(userId), { params });
            set({ loading: false });
        } catch (error: any) {
            set({ error: error.message, loading: false });
        }
    },
    downloadCsvExport: async (filePath) => {
        set({ loading: true, error: null });
        try {
            const response = await axiosInstance.get(FuelSupplierEndpoints.DOWNLOAD_EXPORT, { params: { filePath }, responseType: 'blob' });
            set({ loading: false });
        } catch (error: any) {
            set({ error: error.message, loading: false });
        }
    },
}));
