"use client"

import React, { useEffect } from "react"
import { Box, Stack, Typography } from "@mui/material"
import { ModeType, TableColumn, TableType } from "@/shared/types/common"
import TableControlBar from "@/shared/components/table/table-control-bar"
import TableManager from "@/shared/components/table/table-manager"
import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import type { AbsenceTableRowProps, IAbsence } from "@/shared/types/absence"
import TabsCustom from "../tabs/tabs-custom"
import { TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { IDriverAbsence } from "@/shared/types/driver-absence"
import { IRHAbsence, RHAbsenceTableRowProps } from "@/shared/types/absence-rh"

interface GenericAbsenceViewProps<T extends IAbsence | IDriverAbsence | IRHAbsence> {
    title: string
    tabType: string
    initialData: T[]
    defaultData: T
    defaultTableHead: any[]
    AbsenceTableRow: React.ComponentType<AbsenceTableRowProps<T> | RHAbsenceTableRowProps<T>>
    AbsenceForm: React.ComponentType<{ absence: T;mode: ModeType;onSave: (updatedAbsence: T) => void;onEdit: (updatedAbsence: T) => void;onClose: () => void; isTraiteAbsence?:boolean;   tableHead: TableColumn[];}>
    isTraiteAbsence: boolean
    onTabChange: (activeTab: string) => void
  }
  
  export default function GenericAbsenceView<T extends IAbsence | IDriverAbsence | IRHAbsence>({
    title,
    tabType,
    initialData,
    defaultTableHead,
    AbsenceTableRow,
    AbsenceForm,
    isTraiteAbsence,
    defaultData,
    onTabChange
  }: GenericAbsenceViewProps<T>) {
      const absenceConfig: TabConfig<T> = {
        type: 'absence',
        listTitle: 'Liste absence',
        newItemTitle: 'Nouvelle absence',
        initialData: defaultData,
        getItemCode: (absence) => absence.id,
      };
  const { tabs, activeTab, handleTabClose, handleTabChange, handleTabAdd, handleEdit, handleView, handleCancel } =
    useTabsManager<T>(absenceConfig)

  const absenceUsagerConfig: TableConfig<T> = {
    initialData,
    defaultTableHead,
  }

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
    dataToExport,
  } = useTableManager<T>(absenceUsagerConfig)

 const activeTabData = React.useMemo(() => {
  return tabs.find((tab) => tab.id === activeTab)
}, [tabs, activeTab])

useEffect(() => {
  onTabChange(activeTab)
}, [activeTab, onTabChange])

const renderRow = React.useCallback((row: T) => (
  <AbsenceTableRow
    key={row.id}
    dense={table.dense}
    row={row}
    handleEdit={handleEdit}
    handleDetails={handleView}
    selected={table.selected.includes(row.id)}
    onSelectRow={() => table.onSelectRow(row.id)}
    columns={tableHead}
  />
), [AbsenceTableRow, handleEdit, handleView, table, tableHead])

  return (
    <Box>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="space-between" sx={{ mb: 1 }}>
        <Typography
          variant="h5"
          sx={{
            color: "primary.main",
            fontWeight: (theme) => theme.typography.fontWeightBold,
          }}
        >
          {title}
        </Typography>
        <TableControlBar
          type={TableType.AbsenceUsager}
          activeTab={activeTab}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
        />
      </Stack>
      <TabsCustom
        type={tabType}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      {activeTab === "list" ? (
        <TableManager
          filteredData={filteredData}
          table={table}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
          fixedHeight="calc(100vh - 380px)"
        />
      ) : (
        <AbsenceForm
          absence={activeTabData?.content as T}
          mode={activeTabData?.mode || "view"}
          onSave={handleSave}
          onClose={() => handleCancel(activeTab)}
          onEdit={handleEdit}
          isTraiteAbsence={isTraiteAbsence}
          tableHead={tableHead.slice(0, -1)}
        />
      )}
    </Box>
  )
}

