import { Dialog, DialogTitle, DialogContent, DialogActions, Button, alpha } from '@mui/material';
import { ColumnButton, mainColor, outlinedColor } from '../table/styles';
import ConditionalComponent from '../table/ConditionalComponent';


export interface ConfirmDialogProps {
  open: boolean;
  title: string;
  content: string;
  onConfirm: () => void;
  onCancel: () => void;
  onClose?: () => void;
  confirmButtonText?: string;  
  cancelButtonText?: string;
  confirmButtonColor?: 'success' | 'error' | 'primary' | 'warning';
  useThemeButtons?: boolean; 
  confirmDisabled?: boolean;
}

export default function ConfirmDialog({
  open,
  title,
  content,
  onConfirm,
  onCancel,
  onClose,
  confirmButtonText = 'Confirmer', 
  cancelButtonText = 'Annuler',     
  confirmButtonColor = 'error',
  useThemeButtons = false,
  confirmDisabled = false,
}: ConfirmDialogProps) {
  return (
    <Dialog
      sx={{
        '& .MuiPaper-root': {
          borderRadius: '14px',
        },
        bgcolor:(theme)=>alpha(theme.palette.text.primary,0.5)
      }}
      fullWidth
      maxWidth="xs"
      open={open}
      onClose={!!onClose ? onClose : onCancel}
    >
      <DialogTitle sx={{ fontSize: '1.1rem', fontWeight: 600 }}>{title}</DialogTitle>
      {content && <DialogContent sx={{ typography: 'body2' }}>{content}</DialogContent>}
    
      <DialogActions sx={{ mb: 2, mr: 2, gap: 1 }}>
        <ConditionalComponent
          isValid={useThemeButtons}
          defaultComponent={
            <>
              <ColumnButton 
                onClick={onConfirm} 
                variant="contained" 
                color={confirmButtonColor}
                disabled={confirmDisabled}
              >
                {confirmButtonText} 
              </ColumnButton>
              <ColumnButton 
                variant="outlined" 
                color="inherit" 
                onClick={onCancel}
              >
                {cancelButtonText} 
              </ColumnButton>
            </>
          }
        >
          <>
            <Button 
              variant="contained"
              onClick={onConfirm}
              sx={{
                boxShadow: 'none',
                textTransform: 'none',
                ...mainColor,
              }}
              disabled={confirmDisabled}
            >
              {confirmButtonText}
            </Button>
            <Button 
              variant="outlined"
              onClick={onCancel}
              sx={{
                textTransform: 'none',
                ...outlinedColor,
              }}
            >
              {cancelButtonText}
            </Button>
          </>
        </ConditionalComponent>
      </DialogActions>

    </Dialog>
  );
}