import React, { useState } from 'react';
import { Dialog, DialogTitle, DialogContent, DialogActions, Button, Box, FormControl, InputLabel, Select, MenuItem, Typography } from '@mui/material';
import Upload from '../upload/upload';
import { StyledFormControl, StyledInputLabel, StyledMenuItem, StyledSelect } from '@/shared/theme/css';
import { faFileUpload } from '@fortawesome/free-solid-svg-icons';
import FontAwesome from '../fontawesome';
import ActionButtons from '../form/buttons-action';



interface ImportDialogProps {
  open: boolean;
  onClose: () => void;
  onFileUpload: (file: File, type: string) => void;
  documentTypes: { value: string; label: string }[];
}

const ImportDialog: React.FC<ImportDialogProps> = ({ open, onClose, onFileUpload, documentTypes }) => {
  const [selectedType, setSelectedType] = useState<string>('');

  return (
    <Dialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <DialogTitle
        sx={{
          backgroundColor: "primary.main",
          color: "white",
          fontWeight: "600",
          padding: "16px 24px",
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
        }}
      >
        <Box sx={{ display: "flex", alignItems: "center" }}>
          <FontAwesome
            icon={faFileUpload}
            sx={{ color: "white", marginRight: "12px", width: { xs: 32, lg: 32 }, height: { xs: 32, lg: 32 } }}
          />
          <Typography variant="h6" sx={{ fontSize: "1.25rem", fontWeight: "bold", color: "white" }}>
            Importer un fichier
          </Typography>
        </Box>
      </DialogTitle>
      <DialogContent>
        <Box sx={{ mt: 2 }}>
          <StyledFormControl fullWidth size="small" required sx={{ mb: 2 }}>
            <StyledInputLabel id="type-label">Type de document</StyledInputLabel>
            <StyledSelect
              labelId="type-label"
              value={selectedType}
              displayEmpty
              onChange={(e) => setSelectedType(e.target.value as string)}
            >
              {documentTypes.map((type) => (
                <StyledMenuItem key={type.value} value={type.value}>
                  {type.label}
                </StyledMenuItem>
              ))}
            </StyledSelect>
          </StyledFormControl>
          <Upload
            multiple={false}
            onDrop={(acceptedFiles: File[]) => {
              if (acceptedFiles.length > 0 && selectedType) {
                onFileUpload(acceptedFiles[0], selectedType);
              }
            }}
            accept={{
              'application/pdf': ['.pdf'],
              'application/msword': ['.doc'],
              'application/vnd.openxmlformats-officedocument.wordprocessingml.document': ['.docx'],
              'image/*': ['.png', '.jpg', '.jpeg'],
            }}
            disabled={!selectedType}
          />
        </Box>
      </DialogContent>
      <DialogActions>
      <ActionButtons
        onSave={onClose}
        onCancel={onClose}
        secoundButtonText='Enregistrer'
      />        
      </DialogActions>
    </Dialog>
  );
};

export default ImportDialog;
