"use client"

import type React from "react"

import { Box, Select, MenuItem, type SxProps, type Theme } from "@mui/material"
import { useTheme } from "@mui/material/styles"
import ConfirmDialog from "../dialog/confirmDialog"
import { useState } from "react"

export interface StatusOption {
  value: string
  label: string
  color?: string
  backgroundColor?: string
}

interface StatusSelectProps {
  value: string
  options: StatusOption[]
  onChange: (value: string) => void
  chipComponent: React.ComponentType<any>
  sx?: SxProps<Theme>
  disabled?:boolean
}

export default function StatusMenu({
  value,
  options,
  onChange,
  chipComponent: ChipComponent,
  sx = {},
  disabled
}: StatusSelectProps) {
  const theme = useTheme()
  const [confirmOpen, setConfirmOpen] = useState(false)
  const [pendingValue, setPendingValue] = useState<string | null>(null)

  const currentOption = options.find((opt) => opt.value === value)

  const handleClick = (event: React.MouseEvent) => {
    event.stopPropagation()
  }

  const handleChange = (newValue: string) => {
    if (newValue !== value) {
      setPendingValue(newValue)
      setConfirmOpen(true)
    }
  }

  const handleConfirm = () => {
    if (pendingValue !== null) {
      onChange(pendingValue)
    }
    setPendingValue(null)
    setConfirmOpen(false)
  }

  const handleCancel = () => {
    setPendingValue(null)
    setConfirmOpen(false)
  }

  return (
    <>
      <Box sx={{ display: "inline-flex", alignItems: "center" }}>
        <Select
          value={value || ""}
          onClick={handleClick}
          size="small"
          onChange={(e) => {
            e.stopPropagation()
            handleChange(e.target.value)
          }}
          disabled={disabled}
          renderValue={() => {
            const option = currentOption || options[0]
            return ChipComponent ? (
              <ChipComponent 
                label={option.label} 
                status={option.value} 
                theme={theme} 
                size="small"
                sx={{
                  color: option.color,
                  backgroundColor: option.backgroundColor,
                }}
              />
            ) : null
          }}
          sx={{
            height: "32px",
            width: "140px",
            "& .MuiSelect-select": {
              padding: "4px 32px 4px 0px",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            },
            "& .MuiOutlinedInput-notchedOutline": {
              border: "none",
            },
            ...sx,
          }}
        >
          {options.map((option) => (
            <MenuItem
              key={option.value}
              value={option.value}
              sx={{ display: "flex", justifyContent: "center" }}
            >
              {ChipComponent ? (
                <ChipComponent 
                  label={option.label} 
                  status={option.value} 
                  theme={theme} 
                  size="small"
                  sx={{
                    color: option.color,
                    backgroundColor: option.backgroundColor,
                  }}
                />
              ) : null}
            </MenuItem>
          ))}
        </Select>
      </Box>

      <ConfirmDialog
        open={confirmOpen}
        title="Confirmer le changement de statut"
        content={`Êtes-vous sûr de vouloir changer le statut vers "${options.find(opt => opt.value === pendingValue)?.label}" ?`}
        onConfirm={handleConfirm}
        onCancel={handleCancel}
        confirmButtonColor="primary"
      />
    </>
  )
}

