import * as Yup from 'yup';
import { WorkSchedule, WeeklySchedule, DaySchedule } from '@/shared/types/client';

const dayScheduleSchema = Yup.object().shape({
  id: Yup.string().nullable(),
  isWorkingDay: Yup.boolean().required(),
  startTime: Yup.string().nullable().when('isWorkingDay', {
    is: true,
    then: (schema) => schema.required('L\'heure de début est requise pour un jour travaillé'),
  }),
  endTime: Yup.string().nullable().when('isWorkingDay', {
    is: true,
    then: (schema) => schema.required('L\'heure de fin est requise pour un jour travaillé'),
  }),
  breakStartTime: Yup.string().nullable(),
  breakEndTime: Yup.string().nullable(),
  notes: Yup.string().max(500, 'Les notes ne peuvent pas dépasser 500 caractères'),
  createdAt: Yup.string().nullable(),
  updatedAt: Yup.string().nullable(),
}).test('valid-times', 'L\'heure de fin doit être après l\'heure de début', function(value) {
  if (!value.isWorkingDay || !value.startTime || !value.endTime) {
    return true;
  }
  
  const startTime = new Date(`2000-01-01T${value.startTime}`);
  const endTime = new Date(`2000-01-01T${value.endTime}`);
  
  return endTime > startTime;
}).test('valid-break-times', 'L\'heure de fin de pause doit être après l\'heure de début de pause', function(value) {
  if (!value.breakStartTime || !value.breakEndTime) {
    return true;
  }
  
  const breakStart = new Date(`2000-01-01T${value.breakStartTime}`);
  const breakEnd = new Date(`2000-01-01T${value.breakEndTime}`);
  
  return breakEnd > breakStart;
}).test('break-within-working-hours', 'Les heures de pause doivent être comprises dans les heures de travail', function(value) {
  if (!value.isWorkingDay || !value.startTime || !value.endTime) {
    return true;
  }
  
  if (!value.breakStartTime || !value.breakEndTime) {
    return true;
  }
  
  const startTime = new Date(`2000-01-01T${value.startTime}`);
  const endTime = new Date(`2000-01-01T${value.endTime}`);
  const breakStart = new Date(`2000-01-01T${value.breakStartTime}`);
  const breakEnd = new Date(`2000-01-01T${value.breakEndTime}`);
  
  return breakStart >= startTime && breakEnd <= endTime;
});

const weeklyScheduleSchema = Yup.object().shape({
  id: Yup.string().nullable(),
  monday: dayScheduleSchema,
  tuesday: dayScheduleSchema,
  wednesday: dayScheduleSchema,
  thursday: dayScheduleSchema,
  friday: dayScheduleSchema,
  saturday: dayScheduleSchema,
  sunday: dayScheduleSchema,
  createdAt: Yup.string().nullable(),
  updatedAt: Yup.string().nullable(),
}).test('at-least-one-working-day', 'Au moins un jour de travail doit être défini', function(value) {
  const workingDays = Object.values(value).filter(day => day && typeof day === 'object' && day.isWorkingDay);
  return workingDays.length > 0;
});

export const workScheduleSchema = Yup.object().shape({
  id: Yup.string().optional(),
  collaboratorId: Yup.string().optional(),
  weeklySchedule: weeklyScheduleSchema.required('L\'emploi du temps hebdomadaire est requis'),
  createdAt: Yup.string().optional(),
  updatedAt: Yup.string().optional(),
});

export const validateWorkSchedule = (schedule: WorkSchedule): Promise<void> => {
  return workScheduleSchema.validate(schedule, { abortEarly: false }).then(() => {});
};
