'use client';

import React from 'react';
import { Box, Chip, Tooltip, Typography } from '@mui/material';
import { WorkSchedule } from '@/shared/types/client';
import { AccessTime, Schedule } from '@mui/icons-material';

interface SchedulePreviewProps {
  schedule?: WorkSchedule;
}

const DAYS_SHORT = ['Lun', 'Mar', 'Mer', 'Jeu', 'Ven', 'Sam', 'Dim'];

export default function SchedulePreview({ schedule }: SchedulePreviewProps) {
  if (!schedule) {
    return (
      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
        <Schedule sx={{ fontSize: 16, color: 'text.disabled' }} />
        <Typography variant="body2" color="text.disabled">
          Aucun emploi du temps
        </Typography>
      </Box>
    );
  }

  const workingDays = Object.values(schedule.weeklySchedule).filter(day => day.isWorkingDay).length;
  const totalHours = Object.values(schedule.weeklySchedule).reduce((total, day) => {
    if (day.isWorkingDay && day.startTime && day.endTime) {
      const start = new Date(`2000-01-01T${day.startTime}`);
      const end = new Date(`2000-01-01T${day.endTime}`);
      const diffMs = end.getTime() - start.getTime();
      const diffHours = diffMs / (1000 * 60 * 60);
      return total + diffHours;
    }
    return total;
  }, 0);

  const dayChips = Object.entries(schedule.weeklySchedule).map(([dayKey, day], index) => {
    const isWorking = day.isWorkingDay;
    const dayLabel = DAYS_SHORT[index];
    
    return (
      <Tooltip
        key={dayKey}
        title={
          <Box>
            <Typography variant="body2" fontWeight="bold">
              {dayLabel}
            </Typography>
            {isWorking ? (
              <Box>
                <Typography variant="body2">
                  {day.startTime} - {day.endTime}
                </Typography>
                {day.breakStartTime && day.breakEndTime && (
                  <Typography variant="body2" color="text.secondary">
                    Pause: {day.breakStartTime} - {day.breakEndTime}
                  </Typography>
                )}
                {day.notes && (
                  <Typography variant="body2" color="text.secondary">
                    {day.notes}
                  </Typography>
                )}
              </Box>
            ) : (
              <Typography variant="body2" color="text.secondary">
                Jour libre
              </Typography>
            )}
          </Box>
        }
        arrow
      >
        <Chip
          label={dayLabel}
          size="small"
          color={isWorking ? 'primary' : 'default'}
          variant={isWorking ? 'filled' : 'outlined'}
          sx={{
            minWidth: 32,
            height: 24,
            fontSize: '0.75rem',
            '& .MuiChip-label': {
              px: 0.5,
            },
          }}
        />
      </Tooltip>
    );
  });

  return (
    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
      <AccessTime sx={{ fontSize: 16, color: 'primary.main' }} />
      <Box sx={{ display: 'flex', flexDirection: 'column', gap: 0.5 }}>
        <Box sx={{ display: 'flex', gap: 0.5, flexWrap: 'wrap' }}>
          {dayChips}
        </Box>
        <Typography variant="caption" color="text.secondary">
          {workingDays} jours • {totalHours.toFixed(1)}h
        </Typography>
      </Box>
    </Box>
  );
}
