'use client';

import React from 'react';
import {
  Paper,
  Box,
  Typography,
  FormControlLabel,
  Switch,
  Chip,
  Grid,
  useTheme,
} from '@mui/material';
import { TimePicker } from '@mui/x-date-pickers/TimePicker';
import { parse, isValid } from 'date-fns';
import { AccessTime } from '@mui/icons-material';
import { DaySchedule, WeeklySchedule } from '@/shared/types/client';

interface DayInfo {
  key: string;
  label: string;
}

interface DayScheduleCardProps {
  day: DayInfo;
  daySchedule: DaySchedule;
  isViewMode: boolean;
  onToggle: (dayKey: keyof WeeklySchedule) => void;
  onTimeChange: (
    dayKey: keyof WeeklySchedule,
    field: keyof DaySchedule,
    value: Date | null
  ) => void;
  getFieldError: (dayKey: string, fieldName: string) => string | undefined;
  onFieldBlur: (dayKey: string, fieldName: string) => void;
}

const parseTimeString = (timeString: string | null): Date | null => {
  if (!timeString) return null;
  try {
    const parsed = parse(timeString, 'HH:mm', new Date());
    return isValid(parsed) ? parsed : null;
  } catch {
    return null;
  }
};

export default function DayScheduleCard({
  day,
  daySchedule,
  isViewMode,
  onToggle,
  onTimeChange,
  getFieldError,
  onFieldBlur,
}: DayScheduleCardProps) {
  const theme = useTheme();
  const isWorkingDay = daySchedule.isWorkingDay;

  return (
    <Paper
      elevation={isWorkingDay ? 2 : 0}
      sx={{
        p: 2,
        borderRadius: 2,
        border: `1px solid ${
          isWorkingDay ? theme.palette.primary.main : theme.palette.grey[300]
        }`,
        backgroundColor: isWorkingDay
          ? theme.palette.primary.light
          : theme.palette.grey[50],
        transition: 'all 0.3s ease',
        '&:hover': {
          transform: 'translateY(-2px)',
          boxShadow: theme.shadows[4],
        },
        height: '100%',
      }}
    >
      <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
        <Typography
          variant="h6"
          fontWeight="600"
          sx={{
            color: isWorkingDay
              ? theme.palette.primary.main
              : theme.palette.text.secondary,
            minWidth: 80,
          }}
        >
          {day.label}
        </Typography>

        <FormControlLabel
          control={
            <Switch
              checked={isWorkingDay}
              onChange={() => onToggle(day.key as keyof WeeklySchedule)}
              disabled={isViewMode}
              color="primary"
            />
          }
          label={
            <Chip
              label={isWorkingDay ? 'Jour travaillé' : 'Jour libre'}
              color={isWorkingDay ? 'primary' : 'default'}
              size="small"
              icon={isWorkingDay ? <AccessTime /> : undefined}
            />
          }
          sx={{ ml: 1 }}
        />
      </Box>

      {isWorkingDay && (
        <Grid container spacing={1}>
          <Grid item xs={6}>
            <TimePicker
              label="Début"
              value={parseTimeString(daySchedule.startTime || null)}
              onChange={(value) =>
                onTimeChange(day.key as keyof WeeklySchedule, 'startTime', value)
              }
              disabled={isViewMode}
              slotProps={{
                textField: {
                  fullWidth: true,
                  size: 'small',
                  error: !!getFieldError(day.key, 'startTime'),
                  helperText: getFieldError(day.key, 'startTime'),
                  onBlur: () => onFieldBlur(day.key, 'startTime'),
                },
              }}
            />
          </Grid>
          <Grid item xs={6}>
            <TimePicker
              label="Fin"
              value={parseTimeString(daySchedule.endTime || null)}
              onChange={(value) =>
                onTimeChange(day.key as keyof WeeklySchedule, 'endTime', value)
              }
              disabled={isViewMode}
              slotProps={{
                textField: {
                  fullWidth: true,
                  size: 'small',
                  error: !!getFieldError(day.key, 'endTime'),
                  helperText: getFieldError(day.key, 'endTime'),
                  onBlur: () => onFieldBlur(day.key, 'endTime'),
                },
              }}
            />
          </Grid>
          <Grid item xs={6}>
            <TimePicker
              label="Début pause"
              value={parseTimeString(daySchedule.breakStartTime || null)}
              onChange={(value) =>
                onTimeChange(
                  day.key as keyof WeeklySchedule,
                  'breakStartTime',
                  value
                )
              }
              disabled={isViewMode}
              slotProps={{
                textField: {
                  fullWidth: true,
                  size: 'small',
                  error: !!getFieldError(day.key, 'breakStartTime'),
                  helperText: getFieldError(day.key, 'breakStartTime'),
                  onBlur: () => onFieldBlur(day.key, 'breakStartTime'),
                },
              }}
            />
          </Grid>
          <Grid item xs={6}>
            <TimePicker
              label="Fin pause"
              value={parseTimeString(daySchedule.breakEndTime || null)}
              onChange={(value) =>
                onTimeChange(
                  day.key as keyof WeeklySchedule,
                  'breakEndTime',
                  value
                )
              }
              disabled={isViewMode}
              slotProps={{
                textField: {
                  fullWidth: true,
                  size: 'small',
                  error: !!getFieldError(day.key, 'breakEndTime'),
                  helperText: getFieldError(day.key, 'breakEndTime'),
                  onBlur: () => onFieldBlur(day.key, 'breakEndTime'),
                },
              }}
            />
          </Grid>
        </Grid>
      )}
    </Paper>
  );
}


