import * as Yup from 'yup';
import { INotification, NotificationChannel, NotificationType, NotificationCategory, DeliveryType } from '@/shared/types/notification';

export const notificationSchema = Yup.object().shape({
  title: Yup.string()
    .required('Le titre est obligatoire')
    .min(3, 'Le titre doit contenir au moins 3 caractères')
    .max(100, 'Le titre ne doit pas dépasser 100 caractères'),
  message: Yup.string()
    .required('Le message est obligatoire')
    .min(10, 'Le message doit contenir au moins 10 caractères')
    .max(1000, 'Le message ne doit pas dépasser 1000 caractères'),
  type: Yup.string()
    .required('Le type de notification est obligatoire')
    .oneOf(Object.values(NotificationType), 'Type de notification invalide'),
  category: Yup.string()
    .required('La catégorie est obligatoire')
    .oneOf(Object.values(NotificationCategory), 'Catégorie invalide'),
  recipients: Yup.array()
    .of(Yup.string())
    .min(1, 'Au moins un destinataire doit être sélectionné')
    .required('Les destinataires sont obligatoires'),
  deliveryType: Yup.string()
    .required('Le type d\'envoi est obligatoire')
    .oneOf(Object.values(DeliveryType), 'Type d\'envoi invalide'),
  scheduledDate: Yup.string()
    .when('deliveryType', {
      is: DeliveryType.PLANIFIE,
      then: (schema) => schema
        .required('La date planifiée est obligatoire')
        .test('is-future', 'La date doit être dans le futur', (value) => {
          if (!value) return false;
          const date = new Date(value);
          return date > new Date();
        }),
      otherwise: (schema) => schema.notRequired(),
    }),
  scheduledTime: Yup.string()
    .when('deliveryType', {
      is: DeliveryType.PLANIFIE,
      then: (schema) => schema
        .required('L\'heure planifiée est obligatoire')
        .matches(/^([0-1]?[0-9]|2[0-3]):[0-5][0-9]$/, 'Format d\'heure invalide (HH:mm)'),
      otherwise: (schema) => schema.notRequired(),
    }),
  isActive: Yup.boolean()
    .required('Le statut actif est requis'),
  attachment: Yup.string()
    .optional(),
}); 