import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { useTheme } from '@mui/material/styles';
import { Box, CircularProgress } from '@mui/material';
import SvgColor from '@/shared/components/svg-color';
import { ContratStyledRow, getStatusColor, StatusChip } from '../../contract/styles';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { _CIRCUIT_STATUS, getEtatLabel } from '@/shared/_mock/_circuit';
import StatusMenu from '@/shared/components/table/StatusMenu';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { useAmendmentStore } from '@/shared/api/stores/amendmentStore';
import { AmendmentListDTO } from '@/shared/types/amendment';
import { AmendmentCircuit } from '@/shared/types/ammendment-circuit';



type Props = {
  handleEditAvenant: (avenant:AmendmentCircuit) => void;
  handleDetailsAvenant: (avenant:AmendmentCircuit) => void;
  selected: boolean;
  row: AmendmentCircuit;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: AmendmentCircuit, newStatus: string, field: keyof AmendmentCircuit) => void;
  loading?: boolean;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier avenant',
  duplicate: 'Copier avenant',
  active: 'avenant actif',
  inactive: 'avenant inactif',
}

export default function AvenantTableRow({
  dense,
  row,
  handleEditAvenant,
  handleDetailsAvenant,
  selected,
  onSelectRow,
  columns,
  onStatusChange,
  loading,
}: Props) {
  const { cacheRowData } = useAmendmentStore();


  const theme = useTheme();

  const handleRowDoubleClick = () => {
    const amendmentData = row;

    handleDetailsAvenant(row);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'startDate':
      case 'createdAt':
      case 'circuitEndDate':
        return formatDate(value, dateFormat.isoDate);
      case 'actionType':
        const getTypeLabel = (type: string) => {
          switch (type) {
            case 'ADD_PASSENGER':
              return 'Ajouter d\'un passager';
            case 'REMOVE_PASSENGER':
              return 'Supprimer d\'un passager';
            case 'SCHEDULE_CHANGE':
              return 'Modification d\'horaire';
            case 'USER_ADDRESS_CHANGE':
              return 'Changement d\'adresse utilisateur';
            default:
              return type;
          }
        };
        return getTypeLabel(value);
      case 'circuitState':
        const getStatusConfig = (status: string) => {
          switch (status) {
            case 'CONTROLLED':
              return {
                label: 'Contrôlé',
                backgroundColor: '#E9FBF0',
                color: '#36B37E'
              };
            case 'UNCONTROLLED':
              return {
                label: 'Non contrôlé',
                backgroundColor: '#FFE1E1',
                color: '#FF3434'
              };
            case 'ON_HOLD':
              return {
                label: 'En attente',
                backgroundColor: '#FEF8E7',
                color: '#F9B90B'
              };
            case 'UPDATED':
              return {
                label: 'Mis à jour',
                backgroundColor: '#EEF1FF',
                color: '#5176FE'
              };
            default:
              return {
                label: value,
                backgroundColor: '#F4F6F8',
                color: '#637381'
              };
          }
        };

        const statusConfig = getStatusConfig(value);

        return (
          <StatusChip
            label={statusConfig.label}
            status={value}
            theme={theme}
            size="small"
            sx={{
              backgroundColor: statusConfig.backgroundColor,
              color: statusConfig.color,
              fontWeight: 500,
            }}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover
    style={{ cursor: 'pointer' }}>
      {loading ? (
        <TableCell colSpan={columns.length + 1} align="center">
          <Box sx={{ display: 'flex', justifyContent: 'center', py: 2 }}>
            <CircularProgress size={24} />
          </Box>
        </TableCell>
      ) : (
      <>
      {columns.map((column: any, index: number) => (
          <TableCell
            onDoubleClick={handleRowDoubleClick}
            key={column.id}
            align={column.align }
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0.2 : dense ? 0 : 1, ...(index === 0 && {
              position: 'relative',
            }), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ 
                  position: 'absolute', 
                  top: 0, 
                  left: -4, 
                  height: '100%', 
                  color: row.colorCode,
              }}
            />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof AmendmentCircuit])}
          </TableCell>
        ))}

        <TableCell align="center" sx={{ pl: 1, pr: 1, whiteSpace: 'nowrap' }}>
          <TableActionColumn
            onToggle={(checked) => void 0}
            isActive={row.isActive}
            tooltipTitles={TOOLTIP_TITLES}
            disabled={true}
          />
        </TableCell>
      </>
      )}
    </ContratStyledRow>
  );
}
