"use client"

import { useState, useEffect } from 'react'
import { Box, Typography, Table, TableBody, TableCell, TableContainer, TableHead, TableRow, Paper, Chip } from '@mui/material'
import { useTrips } from '@/shared/sections/chauffeur/listeChauffeur/hooks/use-trips'
import type { TripShortDTO } from '@/shared/types/trip'
import ConditionalComponent from '@/shared/components/table/ConditionalComponent'

interface DriverTripsTableProps {
  driverId?: number
}

const DRIVER_TRIPS_HEAD_CELLS = [
  { id: 'tripTitle', label: 'Titre du trajet' },
  { id: 'periodOfDay', label: 'Période' },
  { id: 'sensTrip', label: 'Sens' },
  { id: 'tripType', label: 'Type' },
]

export default function DriverTripsTable({ driverId }: DriverTripsTableProps) {
  const { trips, loading, error, fetchTrips } = useTrips()

  useEffect(() => {
    const numericId = Number(driverId)
    if (Number.isFinite(numericId) && numericId > 0) {
      fetchTrips(numericId)
    }
  }, [driverId, fetchTrips])

  const renderTripTitle = (trip: TripShortDTO) => {
    return (
      <Box>
        <Typography variant="body2" fontWeight="medium">
          {trip.tripTitled || 'N/A'}
        </Typography>
        <Typography variant="caption" color="text.secondary">
          {trip.periodOfDay} - {trip.sensTrip} - {trip.tripType}
        </Typography>
      </Box>
    )
  }

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'completed':
      case 'terminé':
        return 'success'
      case 'in_progress':
      case 'en_cours':
        return 'warning'
      case 'pending':
      case 'en_attente':
        return 'info'
      case 'cancelled':
      case 'annulé':
        return 'error'
      default:
        return 'default'
    }
  }

  return (
    <Box sx={{ mt: 3 }}>
      <Typography variant="h6" gutterBottom>
        Trajets Assignés
      </Typography>
      
      <ConditionalComponent isValid={loading}>
        <Box sx={{ p: 2, textAlign: 'center' }}>
          <Typography>Chargement des trajets...</Typography>
        </Box>
      </ConditionalComponent>

      <ConditionalComponent isValid={!!error}>
        <Box sx={{ p: 2, textAlign: 'center' }}>
          <Typography color="error">Erreur lors du chargement des trajets: {error}</Typography>
        </Box>
      </ConditionalComponent>

      <ConditionalComponent isValid={!driverId}>
        <Box sx={{ p: 2, textAlign: 'center' }}>
          <Typography color="text.secondary">Aucun chauffeur sélectionné</Typography>
        </Box>
      </ConditionalComponent>

      <ConditionalComponent isValid={!!driverId && !loading && !error && (!trips || trips.length === 0)}>
        <Box sx={{ p: 2, textAlign: 'center' }}>
          <Typography color="text.secondary">Aucun trajet assigné à ce chauffeur</Typography>
        </Box>
      </ConditionalComponent>

      <ConditionalComponent isValid={!!driverId && !loading && !error && !!trips && trips.length > 0}>
        <TableContainer component={Paper} sx={{ mt: 2 }}>
          <Table size="small">
            <TableHead>
              <TableRow>
                {DRIVER_TRIPS_HEAD_CELLS.map((headCell) => (
                  <TableCell key={headCell.id} sx={{ fontWeight: 'bold' }}>
                    {headCell.label}
                  </TableCell>
                ))}
              </TableRow>
            </TableHead>
            <TableBody>
              {trips.map((trip: TripShortDTO, index: number) => (
                <TableRow key={trip.id || index} hover>
                  <TableCell>
                    {renderTripTitle(trip)}
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2">
                      {trip.periodOfDay || 'N/A'}
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2">
                      {trip.sensTrip || 'N/A'}
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2">
                      {trip.tripType || 'N/A'}
                    </Typography>
                  </TableCell>
                  
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </TableContainer>
      </ConditionalComponent>
    </Box>
  )
}
