import TableCell from '@mui/material/TableCell';
import type { TableColumn } from '@/shared/types/common';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { DRIVER_STATUS, IDriverType, UserStatus } from '@/shared/types/driver';
import { DriverStatusChip } from '../../planingChauffeur/style';
import { useDriverStore } from '@/shared/api/stores/driverStore';
import { ContratStyledRow, getStatusColor } from '@/shared/sections/contract/styles';

type Props = {
  handleEdit: (driver: IDriverType) => void;
  handleDetails: (driver: IDriverType) => void;
  handleCopy: (driver: IDriverType) => void;
  selected: boolean;
  row: IDriverType;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IDriverType, newStatus: string, field: keyof IDriverType) => void;
};

const TOOLTIP_TITLES = {
  duplicate: 'Copier chauffeur',
  active: 'Chauffeur Actif',
  inactive: 'Chauffeur Inactif',
};

export default function DriverTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  onStatusChange,
}: Props) {
  const toggleActiveDriver = useDriverStore(state => state.toggleActiveDriver);

  const handleToggleActive = async (checked: boolean) => {
    await toggleActiveDriver(row.id);
  };

  const renderCellContent = (columnId: string, value: any, row: IDriverType) => {
    switch (columnId) {
      case 'fuelVolume':
        return <span style={{ textAlign: 'center', display: 'block' }}>{value}</span>;
      case 'driverStatus':
        return (
          <DriverStatusChip status={value} label={DRIVER_STATUS.find(o => o.value === value)?.label || value} theme={undefined} />
        );
      case 'vehicle.vehiculeName':
        return row.vehicle?.vehiculeName || 'N/A';
      case 'vehicle.fuelCardNumber':
        return row.vehicle?.fuelCardNumber || 'N/A';
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover selected={selected} style={{ cursor: 'pointer' }}>
     

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : dense ? 0 : 2.3 }}
          >
            {renderCellContent(column.id, row[column.id as keyof IDriverType], row)}
          </TableCell>
        ))}
      <TableCell sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onDuplicate={() => handleCopy(row)}
          onToggle={handleToggleActive}
          isActive={row.userStatus === UserStatus.ACTIF}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}
