import React from "react";
import { Box, Typography } from "@mui/material";
import { styled } from "@mui/material/styles";
import { DriverAbsence } from "../types";
import { DriverAbsenceType } from "@/shared/api/stores/driverAbsenceStore";

interface AbsenceBlockProps {
    absence: DriverAbsence;
    width: number | string;
    height: number | string;
    top?: number;
    colorDegree?: number;
}

const AbsenceContainer = styled(Box)(({ theme }) => ({
    position: "absolute",
    left: 0,
    borderRadius: theme.shape.borderRadius,
    padding: theme.spacing(0.5),
    overflow: "hidden",
    whiteSpace: "nowrap",
    textOverflow: "ellipsis",
    fontSize: "0.75rem",
    display: "flex",
    flexDirection: "column",
    alignItems: "center",
    justifyContent: "center",
    zIndex: 1,
}));

export const AbsenceBlock: React.FC<AbsenceBlockProps> = ({ absence, width, height, top = 0, colorDegree = 0 }) => {
    const getAbsenceStyle = (): { backgroundColor: string; color: string; border: string } => {
    switch (absence.type) {
        case DriverAbsenceType.AUTHORIZED_ABSENCE:
            return {
                backgroundColor: "#BBDEFB", // Light blue
                color: "#0D47A1", // Dark blue
                border: "1px solid #42A5F5", // Lighter blue
            };
        case DriverAbsenceType.EXCEPTIONAL_REMOTE_WORK:
            return {
                backgroundColor: "#C8E6C9",
                color: "#388E3C", // Dark green
                border: "1px solid #66BB6A", // Lighter green
            };
        case DriverAbsenceType.EXTERNAL_ASSIGNMENT:
            return {
                backgroundColor: "#FFF59D", // Light yellow
                color: "#F57F17", // Dark yellow
                border: "1px solid #FFEB3B", // Lighter yellow
            };
        case DriverAbsenceType.MEDICAL_APPOINTMENT:
            return {
                backgroundColor: "#FFEBEE", // Light pink
                color: "#D32F2F", // Dark red
                border: "1px solid #EF9A9A", // Lighter pink
            };
        case DriverAbsenceType.PROFESSIONAL_TRAINING:
            return {
                backgroundColor: "#D1C4E9", // Light purple
                color: "#7E57C2", // Dark purple
                border: "1px solid #9575CD", // Lighter purple
            };
        case DriverAbsenceType.UNAUTHORIZED_ABSENCE:
            return {
                backgroundColor: "#FFCDD2", // Light red
                color: "#B71C1C", // Dark red
                border: "1px solid #EF5350", // Lighter red
            };
        case DriverAbsenceType.OTHER:
        default:
            return {
                backgroundColor: "#E0E0E0", // Light gray
                color: "#424242", // Dark gray
                border: "1px solid #9E9E9E", // Medium gray
            };
    }
};

    const absenceStyle = getAbsenceStyle();

    const getAbsenceLabel = (type: DriverAbsenceType): string => {
        switch (type) {
            case DriverAbsenceType.AUTHORIZED_ABSENCE:
                return "Absence Autorisée";
            case DriverAbsenceType.UNAUTHORIZED_ABSENCE:
                return "Absence Non Autorisée";
            case DriverAbsenceType.EXCEPTIONAL_REMOTE_WORK:
                return "Télétravail Exceptionnel";
            case DriverAbsenceType.EXTERNAL_ASSIGNMENT:
                return "Mission Externe";
            case DriverAbsenceType.MEDICAL_APPOINTMENT:
                return "Rendez-vous Médical";
            case DriverAbsenceType.PROFESSIONAL_TRAINING:
                return "Formation Professionnelle";
            case DriverAbsenceType.OTHER:
            default:
                return "Autre";
        }
    };

    const label = `${getAbsenceLabel(absence.type)}`;

    return (
        <AbsenceContainer
            sx={{
                width,
                height,
                top,
                ...absenceStyle,
            }}
        >
            <Typography variant="caption" sx={{ fontWeight: 500 }}>
                {label}
            </Typography>
        </AbsenceContainer>
    );
};