"use client"

import type React from "react"
import { useDraggable } from "@dnd-kit/core"
import { Box, IconButton, Typography } from "@mui/material"
import FontAwesome from "@/shared/components/fontawesome"
import { faUserTie, faCar, faGripVertical } from "@fortawesome/free-solid-svg-icons"
import type { DriverTimeSlot } from "../types"

interface DraggableNonAffectedSlotProps {
  slot: DriverTimeSlot
  colorDegree: number
  onMouseEnter: (e: React.MouseEvent<HTMLElement>, slot: DriverTimeSlot) => void
  onMouseLeave: () => void
  onDoubleClick: (slot: DriverTimeSlot) => void
  onCopySlot?: (slot: DriverTimeSlot) => void
}

export function DraggableNonAffectedSlot({
  slot,
  colorDegree,
  onMouseEnter,
  onMouseLeave,
  onDoubleClick,
  onCopySlot
}: DraggableNonAffectedSlotProps) {
  const { attributes, listeners, setNodeRef, transform, isDragging } = useDraggable({
    id: `non-affected-${slot.id}`,
    data: { ...slot, isFromSidebar: true },
  })

  const style: React.CSSProperties = {
    transform: transform ? `translate3d(${transform.x}px, ${transform.y}px, 0)` : undefined,
    opacity: isDragging ? 0.5 : 1,
  }
  return (
    <Box
      ref={setNodeRef}
      style={style}
      onMouseEnter={(e) => onMouseEnter(e, slot)}
      onMouseLeave={onMouseLeave}
      onDoubleClick={() => onDoubleClick(slot)}
      sx={{
        backgroundColor: `${slot.color}${Math.round((colorDegree / 100) * 255).toString(16).padStart(2, '0')}`,
        borderLeft: `4px solid ${slot.color}`,
        borderRadius: 2,
        p: 2,
        transition: 'all 0.2s ease-in-out',
        '&:hover': {
          transform: isDragging ? undefined : 'translateY(-2px)',
          boxShadow: '0 4px 12px rgba(0,0,0,0.15)',
        },
        '&:active': {
          transform: 'scale(0.98)',
        },
      }}
      {...listeners}
      {...attributes}
    >
      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 1 }}>
        <Typography variant="subtitle2" sx={{ fontWeight: 600, color: 'text.primary' }}>
          {slot.startHeure}:{slot.startMinute} - {slot.endHeure}:{slot.endMinute}
        </Typography>
        <Box sx={{ display: 'flex', gap: 0.5 }}>
          {onCopySlot && (
            <IconButton
              size="small"
              onClick={(e) => {
                e.stopPropagation()
                onCopySlot(slot)
              }}
              sx={{
                width: 24,
                height: 24,
                color: 'text.secondary',
                '&:hover': { color: 'primary.main' }
              }}
            >
              <FontAwesome icon={faCar} width={12} />
            </IconButton>
          )}
          <Box
            sx={{
              width: 24,
              height: 24,
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              color: 'text.secondary',
              '&:hover': { color: 'primary.main' }
            }}
          >
            <FontAwesome icon={faGripVertical} width={12} />
          </Box>
        </Box>
      </Box>

      <Box sx={{ mb: 1 }}>
        <Typography variant="caption" sx={{ color: 'text.primary', fontWeight: 500, display: 'block' }}>
          {slot.name}
        </Typography>
      </Box>

      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mt: 1 }}>
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
          <FontAwesome icon={faUserTie} width={10} color="text.secondary" />
          <Typography variant="caption" sx={{ color: 'text.secondary', fontSize: '0.7rem' }}>
            Pickup: {slot.pickupStartHeure}:{slot.pickupStartMinute}
          </Typography>
        </Box>
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
          <FontAwesome icon={faCar} width={10} color="text.secondary" />
          <Typography variant="caption" sx={{ color: 'text.secondary', fontSize: '0.7rem' }}>
            Return: {slot.returnEndHeure}:{slot.returnEndMinute}
          </Typography>
        </Box>
      </Box>
    </Box>
  )
}