import { Stack, MenuItem } from '@mui/material';
import {
  RoadMapStyledDatePicker,
  RoadMapStyledTimePicker,
  StyledTextField,
} from '@/shared/theme/css';
import FontAwesome from '@/shared/components/fontawesome';
import { faClock } from '@fortawesome/free-solid-svg-icons';
import { CustomDateIcon } from '@/shared/components/form/form-field';
import { FormikProps } from 'formik';
import { SendRoadMapPlanningFormValues } from '@/shared/types/roadMap';

type Props = {
  formik: FormikProps<SendRoadMapPlanningFormValues>;
};

export default function WeeklyFormSection({ formik }: Props) {
  const { values, setFieldValue, errors, touched } = formik;

  return (
    <>
      <Stack direction="row" spacing={2}>
        <RoadMapStyledDatePicker
          value={formik.values.sendDate}
          onChange={(val) => formik.setFieldValue('sendDate', val)}
          views={['year', 'month', 'day']}
          slots={{ openPickerIcon: CustomDateIcon }}
          slotProps={{
            textField: {
              placeholder: 'Démarrage le',
              InputLabelProps: { shrink: true },
              sx: { height: '10%', display: 'flex', alignItems: 'center' },
              error: !!(touched.sendDate && errors.sendDate),
              helperText: touched.sendDate && errors.sendDate,
            },
          }}
          sx={{ width: '100%' }}
        />
        <StyledTextField
          select
          fullWidth
          label="Jour d'envoi"
          value={formik.values.dayOfWeek}
          onChange={(e) => formik.setFieldValue('dayOfWeek', e.target.value)}
          InputProps={{
            sx: { height: '80%', display: 'flex', alignItems: 'center' },
          }}
        >
          {['Lundi', 'Mardi', 'Mercredi', 'Jeudi', 'Vendredi', 'Samedi', 'Dimanche'].map((day) => (
            <MenuItem key={day} value={day}>
              {day}
            </MenuItem>
          ))}
        </StyledTextField>
        <RoadMapStyledTimePicker
          value={formik.values.sendHour}
          onChange={(val) => formik.setFieldValue('sendHour', val)}
          label="Heure d'envoi"
          format="HH:mm"
          views={['hours', 'minutes']}
          slots={{
            openPickerIcon: () => <FontAwesome icon={faClock} sx={{ color: 'primary.main' }} />,
          }}
          slotProps={{
            textField: {
              placeholder: "Heure d'envoi",
              fullWidth: true,
              error: !!(touched.sendHour && errors.sendHour),
              helperText: touched.sendHour && errors.sendHour,
            },
          }}
          sx={{ width: '100%' }}
        />
      </Stack>

      <Stack direction="row" spacing={2} mt={2}>
        <RoadMapStyledDatePicker
          value={formik.values.startDate}
          onChange={(val) => formik.setFieldValue('startDate', val)}
          views={['year', 'month', 'day']}
          slots={{ openPickerIcon: CustomDateIcon }}
          slotProps={{
            textField: {
              placeholder: 'Période du',
              InputLabelProps: { shrink: true },
              fullWidth: true,
              error: !!(touched.startDate && errors.startDate),
              helperText: touched.startDate && errors.startDate,
            },
          }}
          sx={{ width: '100%' }}
        />
        <RoadMapStyledDatePicker
          value={formik.values.endDate}
          onChange={(val) => formik.setFieldValue('endDate', val)}
          views={['year', 'month', 'day']}
          slots={{ openPickerIcon: CustomDateIcon }}
          slotProps={{
            textField: {
              placeholder: 'Au',
              InputLabelProps: { shrink: true },
              fullWidth: true,
              error: !!(touched.endDate && errors.endDate),
              helperText: touched.endDate && errors.endDate,
            },
          }}
          sx={{ width: '100%' }}
        />
      </Stack>
    </>
  );
}
