import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import IconButton from '@mui/material/IconButton';
import type { TableColumn } from '@/shared/types/common';
import { ContratStyledRow, getStatusColor, StatusChip } from '@/shared/sections/contract/styles';
import { IRoadmap, ROADMAP_STATUS_OPTIONS } from '@/shared/types/roadMap';

import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { ActionsIconButton } from '@/shared/theme/css';
import FontAwesome from '@/shared/components/fontawesome';
import { faRocketchat } from '@fortawesome/free-brands-svg-icons';
import SvgColor from '@/shared/components/svg-color';
import StatusMenu from '@/shared/components/table/StatusMenu';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { dateFormat, formatDate } from '@/utils/format-time';


type Props = {
  handleEdit: (roadmap: IRoadmap) => void;
  handleDetails: (roadmap: IRoadmap) => void;
  handleCopy: (roadmap: IRoadmap) => void;
  selected: boolean;
  row: IRoadmap;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IRoadmap, newStatus: string, field: keyof IRoadmap) => void;
  onClickOldRoadMaps : (roadmap: IRoadmap) => void;
  onClickContact: (roadmap: IRoadmap) => void;
};

export default function RoadmapTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  onStatusChange,
  onClickOldRoadMaps,
  onClickContact,
}: Props) {

  const renderCellContent = (column: any, value: any) => {
    
    switch (column.id) {
      case 'date':
        return formatDate(value, dateFormat.dateMonthYear);
      case 'status':
        return (
          <StatusMenu
            value={value}
            options={ROADMAP_STATUS_OPTIONS}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'status')}
            chipComponent={StatusChip}
            sx={{ width: "120px" }}
          />
        );
        case 'contacter':
          return (
            <CustomTooltip title={column.label} arrow>
              <IconButton
                sx={{ py: 0, px: 0.5, ...ActionsIconButton }}
                onClick={(e) => {
                  e.stopPropagation();
                  onClickContact(row);
                }}
              >
                <FontAwesome icon={faRocketchat} width={16} />
              </IconButton>
            </CustomTooltip>
          );
      case 'anciennesListes':
        return (
          <CustomTooltip title={column.label} arrow>
            <IconButton
              sx={{ py: 0, px: 0.5, ...ActionsIconButton }}
              onClick={(e) => {
                e.stopPropagation();
                onClickOldRoadMaps(row);
              }}
            >
              {column.id === 'contacter' ? (
                <FontAwesome icon={faRocketchat} width={16} />
              ) : (
                <SvgColor
                  src="/assets/icons/navbar/ic_table.svg"
                  sx={{
                    width: 16,
                  }}
                />
              )}
            </IconButton>
          </CustomTooltip>
        ); 
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
      {columns.map((column: any, index: number) => (
        <TableCell
          onDoubleClick={() => handleDetails(row)}
          key={column.id}
          align={column.align}
          sx={{
            whiteSpace: 'nowrap',
            ...(index === 0 && {
              position: 'relative',
            }),
            py: dense ? 0 : 1.3,
          }}
        >
          <ConditionalComponent isValid={index === 0}>
            <SvgColor
              src="/assets/icons/ic_border.svg"
              sx={{
                position: 'absolute',
                top: 0, 
                left: -4,
                height: '100%',
                color: getStatusColor(row.status),
              }}
            />
          </ConditionalComponent>

          {renderCellContent(column, row[column.id as keyof IRoadmap])}
        </TableCell>
      ))}
    </ContratStyledRow>
  );
}
