'use client';

import ScheduleVerificationTable from '@/shared/components/roadmap/schedule-verification-table';
import {
  _driverScheduleList,
  _geolocationList,
  DEFAULT_DRIVER_SCHEDULE_TABLE_HEAD,
  DEFAULT_GEOLOCATION_TABLE_HEAD,
} from '@/shared/_mock/_roadMap';
import WeeklyVerificationPanel, {
  VerificationTableType,
} from '@/shared/components/roadmap/weekly-verification-panel';
import { useRoadMapStore } from '@/shared/api/stores/roadMapStore';
import { useEffect, useState } from 'react';
import { IGeolocationData, IGeolocationRoute, PlanningData, ValidatedRoadMapArgs } from '@/shared/types/roadMap';

function transformGeolocData(data: IGeolocationData): {
  id: string;
  date: string;
  matin: string;
  midi: string;
  apresMidi: string;
  soir: string;
  routes: { [key: string]: IGeolocationRoute };
}[] {
  const rows = data.days.map((day, index) => {
    const row: any = {
      id: index.toString(),
      date: day.date,
      matin: '—',
      midi: '—',
      apresMidi: '—',
      soir: '—',
      routes: {},
    };

    day.routes.forEach((route) => {
      const hour = parseInt(route.startHour.split(':')[0], 10);
      const timeRange = `${route.startHour} - ${route.endHour}`;
      if (hour < 12) {
        row.matin = timeRange;
        row.routes['matin'] = route;
      } else if (hour < 14) {
        row.midi = timeRange;
        row.routes['midi'] = route;
      } else if (hour < 18) {
        row.apresMidi = timeRange;
        row.routes['apresMidi'] = route;
      } else {
        row.soir = timeRange;
        row.routes['soir'] = route;
      }
    });

    return row;
  });

  return rows;
}

function transformPlanningData(data: PlanningData): {
  id: string;
  date: string;
  matin: string;
  midi: string;
  apresMidi: string;
  soir: string;
}[] {
  const rows = data.dailyPlannings.map((day, index) => {
    const row: any = {
      id: index.toString(),
      date: day.date,
      matin: '—',
      midi: '—',
      apresMidi: '—',
      soir: '—',
    };

    day.tripPlannings.forEach((trip) => {
      const hour = parseInt(trip.startHour.split(':')[0], 10);
      const timeRange = `${trip.startHour} - ${trip.endHour}`;

      if (hour < 12) {
        row.matin = timeRange;
      } else if (hour < 14) {
        row.midi = timeRange;
      } else if (hour < 18) {
        row.apresMidi = timeRange;
      } else {
        row.soir = timeRange;
      }
    });

    return row;
  });

  return rows;
}
type DriverCombinedTableProps = {
  date:string;
  driverId: string;
};

export default function DriverCombinedTable({date, driverId}:DriverCombinedTableProps) {
  const geolocData = useRoadMapStore((state) => state.geolocData);
  const getGeolocData = useRoadMapStore((state) => state.getGeolocData);

  const planningData = useRoadMapStore((state) => state.planningData);
  const getPlanningData = useRoadMapStore((state) => state.getPlanningData);

  const [geolocTableData, setGeolocTableData] = useState<
    {
      id: string;
      date: string;
      matin: string;
      midi: string;
      apresMidi: string;
      soir: string;
    }[]
  >([]);

  const [vsnTableData, setVsnTableData] = useState<
    {
      id: string;
      date: string;
      matin: string;
      midi: string;
      apresMidi: string;
      soir: string;
    }[]
  >([]);

  useEffect(() => {
    const fetchData = async () => {
      await getGeolocData(1, 2025, 6);
      await getPlanningData(driverId, date);
    };
    fetchData();
  }, []);

  useEffect(() => {
    if (geolocData) {
      const transformed = transformGeolocData(geolocData);
      setGeolocTableData(transformed);
    }
  }, [geolocData]);

  useEffect(() => {
    if (planningData) {
      const transformed = transformPlanningData(planningData);
      setVsnTableData(transformed);
    }
  }, [planningData]);

  return (
    <WeeklyVerificationPanel
      title="Vérification du planning de chauffeur et des données de géolocalisation"
      tableConfigs={[
        {
          title: 'Planning VSN',
          columns: DEFAULT_DRIVER_SCHEDULE_TABLE_HEAD,
          data: vsnTableData,
          verificationType: VerificationTableType.vsnschedule,
          component: ScheduleVerificationTable,
        },
        {
          title: 'Géolocalisation',
          columns: DEFAULT_GEOLOCATION_TABLE_HEAD,
          data: geolocTableData,
          verificationType: VerificationTableType.geolocation,
          component: ScheduleVerificationTable,
        },
      ]}
      date={date}
    />
  );
}
