import React, { useEffect } from 'react';
import { Box, Stack, IconButton, Typography } from '@mui/material';
import { faRotate } from '@fortawesome/free-solid-svg-icons';
import { centerFlexColumn, iconButtonStyles } from '@/shared/theme/css';
import FontAwesome from '@/shared/components/fontawesome';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import {
    _TRAJETS,
    INITIAL_CIRCUIT_TRAJET_DATA,
} from '@/shared/_mock/_circuitTrajets';
import TableManager from '@/shared/components/table/table-manager';
import TabsCustom from '@/shared/components/tabs/tabs-custom';
import TrajetForm from '@/shared/sections/trajet/view/trajet-new-view';
import ExportButton from '@/shared/components/table/table-export-button';
import { TableType } from '@/shared/types/common';
import { TripCircuitDetails, TripCircuitDTO } from '@/shared/types/circuit';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { TabConfig, useTabsManager } from '@/hooks/use-tabs-manager';
import TrajetTableRow from '../trajet-table-row';
import { centerFlexRow, StyledCircularProgress, StyledCircularProgressCentred } from '@/shared/components/form/style';
import { useCustomizedTripsTable } from '../hooks/use-customized-trips-table';

export const trajetTabsConfig: TabConfig<TripCircuitDetails> = {
    type: 'trajet',
    listTitle: 'Liste des trajets personnalisé',
    newItemTitle: 'Nouveau trajet',
    initialData: INITIAL_CIRCUIT_TRAJET_DATA,
    getItemCode: (trip) => trip.id,
};
interface CustomizedTripsFormProps {
    circuitId: string
}
export default function CustomizedTripsList({ circuitId }: CustomizedTripsFormProps) {
    const {
        table,
        filteredData,
        filters,
        tableHead,
        notFound,
        handleFilterChange,
        handleResetFilters,
        dataToExport,
        loading,
        getTripCircuit,
        loadingDetails,
    } = useCustomizedTripsTable(circuitId);
    const {
        tabs,
        activeTab,
        handleTabClose,
        handleTabChange,
        handleView,
        handleCancel,
    } = useTabsManager<TripCircuitDetails>(trajetTabsConfig);

    const activeTabData = React.useMemo(() => {
        return tabs.find((tab) => tab.id === activeTab);
    }, [tabs, activeTab]);

    const renderRow = (row: TripCircuitDTO) => (
        <TrajetTableRow
            dense={table.dense}
            key={row.id}
            row={row}
            handleEditContract={() => void 0}
            handleDetailsContract={() => getTripCircuit(row.id, handleView)}
            columns={tableHead}
        />
    );

    return (
        <Box >
            <Stack
                sx={{
                    visibility: activeTabData?.id === 'list' ? 'visible' : 'hidden',
                }}
                direction="row"
                alignItems="center"
                justifyContent="flex-end"
                gap="8px"
                mt={2}
            >
                <CustomTooltip title="Refresh" arrow>
                    <IconButton onClick={handleResetFilters} sx={iconButtonStyles}>
                        <FontAwesome icon={faRotate} width={16} />
                    </IconButton>
                </CustomTooltip>
            </Stack>
            <TabsCustom
                type="trajet"
                tabs={tabs}
                activeTab={activeTab}
                handleTabChange={handleTabChange}
                handleTabClose={handleTabClose}
            />
            <ConditionalComponent isValid={activeTab === 'list'}
                defaultComponent={
                    <TrajetForm
                        key={activeTabData?.id}
                        trajet={activeTabData?.content as TripCircuitDetails}
                        mode={activeTabData?.mode || 'view'}
                        onClose={() => handleCancel(activeTab)}
                        hasBorder
                        tableHead={tableHead}
                        handleView={handleView}
                        setDrawerMode={()=>{}}
                    />
                }
            >
                <>
                    <ConditionalComponent isValid={!loadingDetails}
                        defaultComponent={
                            <Box sx={centerFlexRow}>
                                <StyledCircularProgress />
                            </Box>
                        }>
                        <>
                            <TableManager
                                filteredData={filteredData}
                                table={table}
                                tableHead={tableHead}
                                notFound={notFound}
                                filters={filters}
                                onFilterChange={handleFilterChange}
                                renderRow={renderRow}
                                sx={{ backgroundColor: 'primary.lighter' }}
                                isTrajetTable={true}
                                isWithoutTabs
                                loading={loading}
                            />
                        </>

                    </ConditionalComponent>
                </>
            </ConditionalComponent>

        </Box>
    );
}
