import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { IHolidayImport } from '@/shared/types/establishment';
import { DrawerStyledRow, getStatusColor } from '@/shared/sections/contract/styles';
import { dateFormat, formatDate } from '@/utils/format-time';
import ConditionalComponent from '../../../../components/table/ConditionalComponent';
import { Box } from '@mui/system';

type Props = {
  handleEdit: (vacance: IHolidayImport) => void;
  handleDetails: (vacance: IHolidayImport) => void;
  handleCopy: (vacance: IHolidayImport) => void;
  selected: boolean;
  row: IHolidayImport;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  isLastCreated: boolean;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier période de vacances',
  duplicate: 'Copier période de vacances',
  active: 'période de vacances actif',
  inactive: 'période de vacances inactif',
};

export default function HolidayImportTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  isLastCreated,
}: Props) {
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "createdAt":
        return formatDate(value, dateFormat.isoDate);
      default:
        return value;
    }
  };

  return (
    <DrawerStyledRow selected={selected} style={{ cursor: 'pointer' }}>
      {columns.filter((column) => column.id !== 'isActive')
        .map((column: any, index) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: dense ? 0 : 2, ...(index === 0 && { position: "relative" }) }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: 'absolute',
                  top: 0,
                  left: -4,
                  height: '100%',
                  color: getStatusColor(row.holidayImportName),
                }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IHolidayImport])}
          </TableCell>
        ))}

      <TableCell sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <ConditionalComponent isValid={isLastCreated}>
          <TableActionColumn
            handleEdit={() => handleEdit(row)}
            tooltipTitles={TOOLTIP_TITLES}
          />
        </ConditionalComponent>
        <ConditionalComponent isValid={!isLastCreated}>
          <Box width={'100%'}></Box>
        </ConditionalComponent>
      </TableCell>
    </DrawerStyledRow>
  );
}