"use client"

import { useEffect, useMemo } from "react"
import { Stack, Typography } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import type { AgencyList } from "@/shared/types/infrastructure"
import { type TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { TableType } from "@/shared/types/common"
import TabsCustom from "@/shared/components/tabs/tabs-custom"
import TableManager from "@/shared/components/table/table-manager"
import AgencyTableRow from "./agency-row"
import AgencyForm from "./agency-form"
import InfrastructureConditionalComponent from "@/shared/components/table/InfrastructureConditionalComponent"
import { useAgencyTable } from "../hooks/use-agency-table"
import { UnsavedChangesDialog } from "@/shared/components/dialog/UnsavedChangesDialog"
import { useSnackbar } from "notistack"


export const INITIAL_AGENCE_DATA: AgencyList = {
  id: '',
  name: '',
  address: '',
  description: '',
  createdAt: new Date().toISOString().split('T')[0],
  active: true,
};

const agencyTabsConfig: TabConfig<AgencyList> = {
  type: TableType.Agence,
  listTitle: "Liste agences",
  newItemTitle: "Nouvelle agence",
  initialData: INITIAL_AGENCE_DATA,
  getItemCode: (agency) => agency.name,
}

export default function AgencyListView() {

  const { enqueueSnackbar } = useSnackbar();

  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
    markTabAsSaved,
  } = useTabsManager<AgencyList>(agencyTabsConfig)

  const {
    error,
    loading,
    clearError,
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    onPageChange,
    page,
    onRowsPerPageChange,
    rowsPerPage,
    totalAgencies,
    sortBy,
    sortDirection,
    onResetFilters
  } = useAgencyTable()

  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab)
  }, [tabs, activeTab])

  const renderRow = (row: AgencyList) => (
    <AgencyTableRow
      key={row.id}
      dense={table.dense}
      row={row}
      handleEdit={handleEdit}
      handleCopy={handleCopy}
      handleDetails={handleView}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)}
      handleOpenConsult={handleView}
    />
  )

  const isList = activeTab === "list"

  useEffect(() => {
    if (error) {
      enqueueSnackbar(error, { variant: 'error' })
      clearError()
    }
  }, [error, enqueueSnackbar, clearError])

  return (
    <>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.Agence}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={onResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={filteredData}
        />
      </Stack>

    

      <TabsCustom
        type={TableType.Agence}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      <InfrastructureConditionalComponent isValid={isList}
        defaultComponent={
          <AgencyForm
            agency={activeTabData?.content as AgencyList}
            onEdit={handleEdit}
            onClose={(force) => handleCancel(activeTab, force)}
            mode={activeTabData?.mode || "add"}
            tableHead={tableHead}
            updateTabContent={updateTabContent}
            tabId={activeTab}
            markTabAsSaved={markTabAsSaved}
            tabContent={activeTabData?.content as AgencyList}
          />
        }
      >
        <>
          <TableManager
            loading={loading}
            filteredData={filteredData}
            table={table}
            tableHead={tableHead}
            notFound={notFound}
            filters={filters}
            onFilterChange={handleFilterChange}
            renderRow={renderRow}
            fixedHeight="calc(100vh - 320px)"
            onPageChange={onPageChange}
            onRowsPerPageChange={onRowsPerPageChange}
            rowsPerPage={rowsPerPage || 20}
            page={page || 0}
            count={totalAgencies}
            sortBy={sortBy || 'id'}
            sortDirection={sortDirection || 'desc'}
          />
        </>
      </InfrastructureConditionalComponent>
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={() => handleConfirmDialogAction()}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </>
  )
}
