import { Box, Tooltip } from "@mui/material";
import CloseIcon from '@mui/icons-material/Close';
import CheckIcon from '@mui/icons-material/Check';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface ColorOptionProps {
    color: { name: string, hex: string };
    selected: boolean;
    disabled?: boolean;
    onSelect: (hexColor: string) => void;
}

export default function ColorOption({ color, selected, disabled = false, onSelect }: ColorOptionProps) {
    const handleClick = (e: React.MouseEvent) => {
        if (disabled) {
            e.preventDefault();
            e.stopPropagation();
            return;
        }
        onSelect(color.hex);
    };

  return (
    <Tooltip 
      title={
        disabled 
          ? `${color.name} - Déjà utilisée` 
          : selected 
              ? `${color.name} - Sélectionnée` 
              : color.name
      } 
      placement="top" 
      arrow
    >
      <Box
        component={disabled ? 'div' : 'button'}
        type={disabled ? undefined : 'button'}
        role="radio"
        aria-checked={selected}
        onClick={handleClick}
        aria-label={`Choisir la couleur ${color.name}${disabled ? ' (déjà utilisée)' : ''}`}
        disabled={disabled}
        sx={{
          width: 28,
          height: 28,
          borderRadius: 1.5,
          background: color.hex,
          border: selected ? '3px solid #1976d2' : '2px solid #e0e0e0',
          outline: 'none',
          cursor: disabled ? 'not-allowed' : 'pointer',
          opacity: disabled ? 0.25 : 1,
          boxShadow: selected ? '0 2px 6px rgba(25,118,210,0.2)' : '0 1px 3px rgba(0,0,0,0.1)',
          transition: 'all 0.2s ease',
          padding: 0,
          position: 'relative',
          overflow: 'visible',
          pointerEvents: disabled ? 'none' : 'auto',
          filter: disabled ? 'grayscale(100%)' : 'none',
          '&:hover': disabled ? {} : {
            boxShadow: selected
              ? '0 4px 8px rgba(25,118,210,0.3)'
              : '0 2px 6px rgba(0,0,0,0.15)',
            transform: 'scale(1.1)',
          },
          '&:disabled': {
            cursor: 'not-allowed',
            opacity: 0.35,
            pointerEvents: 'none',
          }
        }}
      >
                <ConditionalComponent isValid={disabled}>
                    <CloseIcon
                        sx={{
                            position: 'absolute',
                            top: '50%',
                            left: '50%',
                            transform: 'translate(-50%, -50%)',
                            fontSize: '18px',
                            color: '#ff1744',
                            fontWeight: 'bold',
                            textShadow: '0 0 2px rgba(255,255,255,0.8)',
                            filter: 'drop-shadow(0 1px 2px rgba(0,0,0,0.3))',
                        }}
                    />
                </ConditionalComponent>
                <ConditionalComponent isValid={selected && !disabled}>
                    <CheckIcon
                        sx={{
                            position: 'absolute',
                            top: '50%',
                            left: '50%',
                            transform: 'translate(-50%, -50%)',
                            fontSize: '18px',
                            color: '#fff',
                            fontWeight: 'bold',
                            textShadow: '0 0 2px rgba(0,0,0,0.5)',
                            filter: 'drop-shadow(0 1px 2px rgba(0,0,0,0.5))',
                        }}
                    />
                </ConditionalComponent>
            </Box>
        </Tooltip>
    );
}