import { useState, useEffect } from 'react';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { TableColumn } from '@/shared/types/common';
import React from 'react';
import type { DepartmentList, DepartmentPageFilterParams } from '@/shared/types/infrastructure';
import { useInfrastructureStore } from '@/shared/api/stores/admin-service/infrastructureStore';

export const DEFAULT_TABLE_HEAD: TableColumn[] = [
    { id: 'id', label: 'N°', type: 'text' },
    { id: "name", label: "Nom", type: "text" },
    { id: "code", label: "Code", type: "text" },
    { id: "address", label: "Adresse", type: "text" },
    { id: "description", label: "Description", type: "text" },
    { id: "createdAt", label: "Date de création", type: "date" },
    { id: "zoneName", label: "Zone", type: "text" },
    { id: "", label: "Action" },
];

const departmentConfig: TableConfig<DepartmentList> = {
    initialData: [],
    defaultTableHead: DEFAULT_TABLE_HEAD,
};

export function useDepartmentTable() {
    const [params, setParams] = useState<DepartmentPageFilterParams>({
        page: 0,
        size: 20,
        sortBy: 'id',
        sortDirection: 'desc'
    });

    const { departmentPage, departmentPageTotalElements, loading, error, fetchDepartmentPage, clearError } = useInfrastructureStore();

    useEffect(() => {
        fetchDepartmentPage(params);
    }, [params]);

    const tableManager = useTableManager<DepartmentList>({
        ...departmentConfig,
        initialData: departmentPage || [],
    });

    const filteredDepartmentPage = React.useMemo(() => {
        if (!tableManager.filters || Object.keys(tableManager.filters).length === 0) {
            return departmentPage || [];
        }

        return (departmentPage || []).filter((item: any) => {
            return Object.entries(tableManager.filters).every(([key, { value, condition }]) => {
                if (!value) return true;
                
                const itemValue = item[key];
                if (!itemValue) return false;

                const stringValue = String(itemValue).toLowerCase();
                const filterValue = String(value).toLowerCase();

                switch (condition) {
                    case 'equals':
                        return stringValue === filterValue;
                    case 'startsWith':
                        return stringValue.startsWith(filterValue);
                    case 'endsWith':
                        return stringValue.endsWith(filterValue);
                    case 'contains':
                    default:
                        return stringValue.includes(filterValue);
                }
            });
        });
    }, [departmentPage, tableManager.filters]);

    const onResetFilters = () => {
        tableManager.handleResetFilters();
        setParams({
            page: 0,
            size: 20,
            sortBy: 'id',
            sortDirection: 'desc'
        });
    };

    const isValidDateInput = (value: any): boolean => {
        if (!value || typeof value !== 'string') return false;
        const date = new Date(value);
        return date instanceof Date && !isNaN(date.getTime());
    };
    
    const isValidFilterValue = (value: any): boolean => {
        return value !== null && value !== undefined && value !== '';
    };

    useEffect(() => {
        const baseParams: DepartmentPageFilterParams = {
            page: 0,
            size: tableManager.table.rowsPerPage || 20,
            sortBy: tableManager.table.orderBy || 'id',
            sortDirection: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
        };

        const filterParams: Record<string, string | number> = {};

        Object.entries(tableManager.filters).forEach(([key, { value, condition }]) => {
            if (key === 'createdAt' && isValidDateInput(value)) {
                const date = new Date(value);
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0');
                const day = String(date.getDate()).padStart(2, '0');
                filterParams[key] = `${year}-${month}-${day}T00:00:00`;
            } else if (isValidFilterValue(value)) {
                const filterKey = condition ? `${key}_${condition}` : key;
                filterParams[filterKey] = value;
            }
        });

        const newParams: DepartmentPageFilterParams = {
            ...baseParams,
            ...filterParams
        };

        const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
        if (paramsChanged) {
            setParams(newParams);
        }
    }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order, tableManager.table.rowsPerPage, params]);

    const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
        setParams(prev => ({
            ...prev,
            page: newPage
        }));
    };

    const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const size = parseInt(event.target.value, 10);
        if (!isNaN(size) && size > 0) {
            setParams(prev => ({
                ...prev,
                size,
                page: 0
            }));
        }
    };

    return {
        ...tableManager,
        loading,
        error,
        clearError,
        totalDepartments: departmentPageTotalElements || 0,
        onPageChange: handlePageChange,
        page: params.page,
        onRowsPerPageChange: handleRowsPerPageChange,
        rowsPerPage: params.size,
        sortBy: params.sortBy || 'id',
        sortDirection: params.sortDirection || 'desc',
        filteredData: filteredDepartmentPage,
        onResetFilters: onResetFilters
    };
}