"use client"

import { useMemo, useEffect } from "react"
import { Stack, Typography } from "@mui/material"
import TableControlBar from "@/shared/components/table/table-control-bar"
import type { DepartmentList } from "@/shared/types/infrastructure"
import { type TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { TableType } from "@/shared/types/common"
import TabsCustom from "@/shared/components/tabs/tabs-custom"
import TableManager from "@/shared/components/table/table-manager"
import DepartmentTableRow from "./department-row"
import DepartmentForm from "./department-form"
import InfrastructureConditionalComponent from "@/shared/components/table/InfrastructureConditionalComponent"
import { UnsavedChangesDialog } from "@/shared/components/dialog/UnsavedChangesDialog"
import { useDepartmentTable } from "../hooks/use-department-table"
import { useSnackbar } from "notistack"

export const INITIAL_DEPARTMENT_DATA: DepartmentList = {
  id: '',
  name: '',
  code: '',
  address: '',
  description: '',
  hexColor: '',
  zoneName: '',
  createdAt: new Date().toISOString().split('T')[0],
  active: true,
};

const departmentTabsConfig: TabConfig<DepartmentList> = {
  type: TableType.Departement,
  listTitle: "Liste départements",
  newItemTitle: "Nouveau département",
  initialData: INITIAL_DEPARTMENT_DATA,
  getItemCode: (department) => department.name,
}

export default function DepartmentListView() {

  const { enqueueSnackbar } = useSnackbar();

  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
    markTabAsSaved,
  } = useTabsManager<DepartmentList>(departmentTabsConfig)

  const {
    error,
    loading,
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    onPageChange,
    page,
    onRowsPerPageChange,
    rowsPerPage,
    totalDepartments,
    sortBy,
    sortDirection,
    onResetFilters,
    clearError
  } = useDepartmentTable()

  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab)
  }, [tabs, activeTab])

  const renderRow = (row: DepartmentList) => (
    <DepartmentTableRow
      key={row.id}
      dense={table.dense}
      row={row}
      handleEdit={handleEdit}
      handleCopy={handleCopy}
      handleDetails={handleView}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)}
      handleOpenConsult={handleView}
    />
  )

  const isList = activeTab === "list"

  useEffect(() => {
    if (error) {
      enqueueSnackbar(error, { variant: 'error' })
      clearError()
    }
  }, [error, enqueueSnackbar, clearError])

  return (
    <>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.Departement}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={onResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={filteredData}
        />
      </Stack>

    

      <TabsCustom
        type={TableType.Departement}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      <InfrastructureConditionalComponent
        isValid={isList}
        defaultComponent={
        <DepartmentForm
          department={activeTabData?.content as DepartmentList}
          onEdit={handleEdit}
          onClose={(force) => handleCancel(activeTab, force)}
          mode={activeTabData?.mode || "add"}
          tableHead={tableHead}
          updateTabContent={updateTabContent}
          tabId={activeTab}
          markTabAsSaved={markTabAsSaved}
          tabContent={activeTabData?.content as DepartmentList}
        />
        }
      >
        <>
          <TableManager
            loading={loading}
            filteredData={filteredData}
            table={table}
            tableHead={tableHead}
            notFound={notFound}
            filters={filters}
            onFilterChange={handleFilterChange}
            renderRow={renderRow}
            fixedHeight="calc(100vh - 320px)"
            onPageChange={onPageChange}
            onRowsPerPageChange={onRowsPerPageChange}
            rowsPerPage={rowsPerPage || 20}
            page={page || 0}
            count={totalDepartments}
            sortBy={sortBy || 'id'}
            sortDirection={sortDirection || 'desc'}
          />
        </>
      </InfrastructureConditionalComponent>
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={() => handleConfirmDialogAction()}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </>
  )
}
