import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { SectorList } from '@/shared/types/infrastructure';
import { ContratStyledRow } from '@/shared/sections/contract/styles';
import { alpha, Box } from '@mui/material';
import { useTheme } from '@mui/material/styles';
import dayjs from 'dayjs';
import { useInfrastructureStore } from '@/shared/api/stores/admin-service/infrastructureStore';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEdit: (sector: SectorList) => void;
  handleDetails: (sector: SectorList) => void;
  handleCopy: (sector: SectorList) => void;
  selected: boolean;
  row: SectorList;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  handleOpenConsult: (sector: SectorList) => void;
};

const TOOLTIP_TITLES = {
  active: 'Secteur activé',
  inactive: 'Secteur désactivé',
};

export default function SectorTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  handleOpenConsult,
}: Props) {
  const theme = useTheme();

  const { toggleSectorActivation } = useInfrastructureStore();

  const handleDoubleClick = () => {
    handleOpenConsult(row);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'createdAt':
        return dayjs(value).format('DD-MM-YYYY HH:mm');
      case 'address':
        return (
          <Box
            sx={{
              maxWidth: '200px',
              overflow: 'hidden',
              textOverflow: 'ellipsis',
              whiteSpace: 'nowrap',
              cursor: 'help',
            }}
            title={value} // Tooltip avec l'adresse complète
          >
            {value}
          </Box>
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow
      hover
      style={{
        backgroundColor: selected
          ? alpha(theme.palette.primary.main, 0.1)
          : undefined,
        cursor: 'pointer',
      }}
    >
      {columns.map((column: any,index:number) => (
        <TableCell
          onDoubleClick={handleDoubleClick}
          key={column.id}
          align={column.align}
          sx={{ 
            whiteSpace: column.id === 'address' ? 'normal' : 'nowrap', 
            py: dense ? 0 : 2,
            ...(column.id === 'address' && {
              maxWidth: '200px',
              width: '200px',
            }),
            ...(index === 0 && {
              position: 'relative',
            })
          }}
        >
          <ConditionalComponent isValid={index === 0}>
            <SvgColor
              src="/assets/icons/ic_border.svg"
              sx={{
                position: 'absolute',
                top: 0,
                left: -4,
                height: '100%',
                color: theme.palette.primary.main,
              }}
            />
          </ConditionalComponent>
          {renderCellContent(column.id, row[column.id as keyof SectorList])}
        </TableCell>
      ))}
      <TableCell sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onToggle={(checked) => toggleSectorActivation(row.id, checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}