import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { useFuelSupplierStore } from '@/shared/api/stores/fleet-service/fuel-supplier-store';
import { TableColumn } from '@/shared/types/common';
import { FuelSupplier } from '@/shared/types/fleet-management/fuelSupplier';
import { useEffect, useState } from 'react';

export const DEFAULT_FUEL_SUPPLIER_TABLE_HEAD: TableColumn[] = [
  { id: "id", label: "N°", type: "text" },
  { id: "supplierName", label: "Pétrolier", type: "text" },
  { id: "fuelType", label: "Type Carburant", type: "text" },
  { id: "", label: "Action" },
];

interface FuelSupplierPageFilterParams {
  page: number;
  size: number;
  orderBy: string;
  orderDir: 'asc' | 'desc';
  [key: string]: any;
}

const fuelSupplierConfig: TableConfig<FuelSupplier> = {
  initialData: [],
  defaultTableHead: DEFAULT_FUEL_SUPPLIER_TABLE_HEAD,
};

export function useFuelSupplierTable() {
  const [params, setParams] = useState<FuelSupplierPageFilterParams>({
    page: 0,
    size: 20,
    orderBy: 'id',
    orderDir: 'desc',
  });

  const {
    fuelSuppliers,
    loading,
    error,
    fetchAllFuelSuppliers,
    startCsvExport
  } = useFuelSupplierStore();

  useEffect(() => {
    fetchAllFuelSuppliers();
  }, [params]);

  const tableManager = useTableManager<FuelSupplier>({
    ...fuelSupplierConfig,
    initialData: fuelSuppliers || [],
  });

  const onResetFilters = () => {
    tableManager.handleResetFilters();
    setParams({
      page: 0,
      size: 20,
      orderBy: 'id',
      orderDir: 'desc'
    });
  };

  useEffect(() => {
    if (tableManager.table.rowsPerPage > 0) {
      setParams(prev => ({
        ...prev,
        page: tableManager.table.page,
        size: tableManager.table.rowsPerPage,
      }));
    }
  }, [tableManager.table.page, tableManager.table.rowsPerPage]);

  const isValidFilterValue = (value: any): boolean => {
    return value !== null && value !== undefined && value !== '';
  };

  useEffect(() => {
    const baseParams: FuelSupplierPageFilterParams = {
      page: tableManager.table.page || 0,
      size: tableManager.table.rowsPerPage || 20,
      orderBy: tableManager.table.orderBy || 'id',
      orderDir: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
    };

    const filterParams: Record<string, string | number> = {};

    Object.entries(tableManager.filters).forEach(([key, { value }]) => {
      if (key === 'id' && isValidFilterValue(value)) {
        filterParams[key] = Number(value);
      } else if (isValidFilterValue(value)) {
        filterParams[key] = value;
      }
    });

    const newParams: FuelSupplierPageFilterParams = {
      ...baseParams,
      ...filterParams
    };

    const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
    if (paramsChanged) {
      setParams(newParams);
    }
  }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order]);

  const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
    setParams(prev => ({
      ...prev,
      page: newPage
    }));
  };

  const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const size = parseInt(event.target.value, 10);
    if (!isNaN(size) && size > 0) {
      setParams(prev => ({
        ...prev,
        size,
        page: 0
      }));
    }
  };

  const handleStartCsvExportFuelSuppliers = () => {
    startCsvExport(params, parseInt(localStorage.getItem("userId") as string));
  };

  return {
    ...tableManager,
    loading,
    error,
    totalFuelSuppliers: fuelSuppliers?.length || 0,
    onPageChange: handlePageChange,
    page: params.page,
    onRowsPerPageChange: handleRowsPerPageChange,
    rowsPerPage: params.size,
    sortBy: params.orderBy || 'id',
    sortDirection: params.orderDir || 'desc',
    filteredData: fuelSuppliers.sort(),
    handleStartCsvExportFuelSuppliers: handleStartCsvExportFuelSuppliers,
    onResetFilters: onResetFilters,
  };
}