'use client';

import React, { useMemo } from 'react';
import { Box, Stack } from '@mui/material';
import { HEADER } from '@/layouts/config-layout';
import TableControlBar from '@/shared/components/table/table-control-bar';
import { useFuelSupplierTabs } from '../hooks/use-fuel-supplier-tabs';
import { useFuelSupplierTable } from '../hooks/use-fuel-supplier-table';
import { TableType } from '@/shared/types/common';
import CustomTabs from '@/shared/components/tabs/tabs-custom';
import TableManager from '@/shared/components/table/table-manager';
import FuelSupplierForm from './fuel-supplier-new-view';
import FuelSupplierTableRow from './fuel-supplier-table-row';
import { UnsavedChangesDialog } from '@/shared/components/dialog/UnsavedChangesDialog';
import { FuelSupplier } from '@/shared/types/fleet-management/fuelSupplier';

export default function FuelSupplierListView() {
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
    setTabHasUnsavedChanges,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
  } = useFuelSupplierTabs();

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
    dataToExport,
    handleStartCsvExportFuelSuppliers
  } = useFuelSupplierTable();
  
  const activeTabData = React.useMemo(() => {
    return tabs.find(tab => tab.id === activeTab);
  }, [tabs, activeTab]);

  const exportOptions = useMemo(() => [
        {
          label: "Exporter en CSV",
          action: () => handleStartCsvExportFuelSuppliers()
        }
      ], [handleStartCsvExportFuelSuppliers]);

  const renderRow = (row: FuelSupplier) => (
    <FuelSupplierTableRow
      key={row.id}
      row={row}
      handleCopyFuelSupplier={handleCopy}
      handleEditFuelSupplier={handleEdit}
      handleDetailsFuelSupplier={handleView}
      selected={table.selected.includes(row.id.toString())}
      onSelectRow={() => table.onSelectRow(row.id.toString())}
      columns={tableHead.slice(0,-1)}
    />
  )

  return (
    <Box
      sx={{
        display: 'flex',
        flexDirection: 'column',
        height: `calc(100vh - ${HEADER.H_DESKTOP + 8}px)`,
      }}
    >
      <Stack
        flexGrow={0}
        direction="row"
        alignItems="center"
        justifyContent="flex-end"
        spacing={{ xs: 0.5, sm: 1 }}
      >
        <TableControlBar
          type={TableType.FuelSupplier}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={handleResetFilters} 
          onResetColumns={handleResetColumns} 
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
          useBackendExport={true}
          exportOptions={exportOptions}
        />
      </Stack>

      <CustomTabs
        type={TableType.FuelSupplier}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      {activeTab === 'list' ? (
        <TableManager
          filteredData={filteredData}
          table={table}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
        />
      ) : (
        <FuelSupplierForm
          key={activeTabData?.id}
          fuelSupplier={activeTabData?.content as FuelSupplier}
          mode={activeTabData?.mode || 'view'} 
          onClose={() => handleCancel(activeTab, true)}
          onEdit={handleEdit}
          tableHead={tableHead}
          setHasUnsavedChanges={(value) => setTabHasUnsavedChanges(activeTab, value)}
          updateTabContent={updateTabContent}
          tabId={activeTab}
        />
      )}
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={() => handleConfirmDialogAction()}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </Box>
  );
}