import React from 'react';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { TableColumn } from '@/shared/types/common';
import { ContratStyledRow } from '@/shared/sections/contract/styles';
import { FuelSupplier } from '@/shared/types/fleet-management/fuelSupplier';
import { useFuelSupplierStore } from '@/shared/api/stores/fleet-service/fuel-supplier-store';
import { Chip } from '@mui/material';

type Props = {
  handleEditFuelSupplier: (fuelSupplier: FuelSupplier) => void;
  handleDetailsFuelSupplier: (fuelSupplier: FuelSupplier) => void;
  handleCopyFuelSupplier: (fuelSupplier: FuelSupplier) => void;
  selected: boolean;
  row: FuelSupplier;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
};

const TOOLTIP_TITLES = {
  edit: 'Modifier pétrolier',
  duplicate: 'Copier pétrolier',
  active: 'Pétrolier actif',
  inactive: 'Pétrolier inactif',
};

export default function FuelSupplierTableRow({
  row,
  handleEditFuelSupplier,
  handleDetailsFuelSupplier,
  handleCopyFuelSupplier,
  selected,
  onSelectRow,
  columns,
}: Props) {

  const { toggleFuelSupplierActive } = useFuelSupplierStore();

  const handleToggleActivation = async () => {
    await toggleFuelSupplierActive(parseInt(row.id));
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'fuelType':
        return row.fuelType?.name || '-';
      case 'isActive':
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
      {columns.map((column: any, index: number) => (
        <TableCell
          onDoubleClick={() => handleDetailsFuelSupplier(row)}
          key={column.id}
          align={column.align}
          sx={{ whiteSpace: 'nowrap' }}
        >
          {renderCellContent(column.id, row[column.id as keyof FuelSupplier])}
        </TableCell>
      ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditFuelSupplier(row)}
          onToggle={() => handleToggleActivation()}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}