import { useState, useEffect } from 'react';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { TableColumn } from '@/shared/types/common';
import React from 'react';
import { FuelType, FuelTypePageFilterParams } from '@/shared/types/fleet-management/fuelType';
import { useFuelTypeStore } from '@/shared/api/stores/fleet-service/fuel-type-store';
import { useSnackbar } from 'notistack';

export const FUEL_TYPE_TABLE_HEAD: TableColumn[] = [
    { id: "id", label: "N°", type: "text" },
    { id: "name", label: "Nom du type", type: "text" },
    { id: "active", label: "Statut", type: "boolean" },
    { id: "actions", label: "Actions" },
];

const fuelTypeConfig: TableConfig<FuelType> = {
    initialData: [],
    defaultTableHead: FUEL_TYPE_TABLE_HEAD,
}

export function useFuelTypeTable() {

    const { enqueueSnackbar } = useSnackbar();

    const [keyword, setKeyword] = useState("");

    const [params, setParams] = useState<FuelTypePageFilterParams>({
        page: 0,
        size: 20,
        orderBy: 'id',
        orderDir: 'desc'
    });

    const {
        loading,
        error,
        fuelTypesPage,
        totalElements,
        fetchFuelTypesPage,
        startExportFiltredFuelTypes
    } = useFuelTypeStore();

    useEffect(() => {
        fetchFuelTypesPage(params);
    }, [params]);

    const tableManager = useTableManager<any>({
        ...fuelTypeConfig,
        initialData: fuelTypesPage || [],
    });

    const onResetFilters = () => {
        tableManager.handleResetFilters();
        setParams({
            page: 0,
            size: 20,
            orderBy: 'id',
            orderDir: 'desc'
        });
    };

    useEffect(() => {
        if (tableManager.table.rowsPerPage > 0) {
            setParams(prev => ({
                ...prev,
                page: tableManager.table.page,
                size: tableManager.table.rowsPerPage,
            }));
        }
    }, [tableManager.table.page, tableManager.table.rowsPerPage]);

    const isValidFilterValue = (value: any): boolean => {
        return value !== null && value !== undefined && value !== '';
    };

    useEffect(() => {
        const baseParams: FuelTypePageFilterParams = {
            page: tableManager.table.page || 0,
            size: tableManager.table.rowsPerPage || 20,
            orderBy: tableManager.table.orderBy || 'id',
            orderDir: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
        };

        const filterParams: Record<string, string | number | null> = {};
        Object.entries(tableManager.filters).forEach(([key, { value }]) => {
            if (key === 'id' && isValidFilterValue(value)) {
                filterParams[key] = Number(value);
            } else if (isValidFilterValue(value)) {
                filterParams[key] = value;
            }
        });

        let newParams: FuelTypePageFilterParams;

        // Si des filtres spécifiques (id ou name) sont actifs, ils ont la priorité sur keyword
        const hasSpecificFilters = Object.keys(filterParams).length > 0;

        if (hasSpecificFilters) {
            newParams = {
                ...baseParams,
                ...filterParams
            };
        } else if (keyword) {
            newParams = {
                ...baseParams,
                keyword: keyword
            };
        } else {
            newParams = baseParams;
        }

        const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
        if (paramsChanged) {
            setParams(newParams);
        }
    }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order, keyword]);

    const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
        setParams(prev => ({
            ...prev,
            page: newPage
        }));
    };

    const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const size = parseInt(event.target.value, 10);
        if (!isNaN(size) && size > 0) {
            setParams(prev => ({
                ...prev,
                size,
                page: 0
            }));
        }
    };

    const userId = localStorage.getItem('userId');

    const handleStartExportFuelTypes = () => {
        if (userId) {
            startExportFiltredFuelTypes(userId, params);
        } else {
            enqueueSnackbar("Nous n'avons pas pu identifier l'utilisateur connecté.");
        }
    };

    return {
        ...tableManager,
        loading,
        error,
        totalFuelTypes: totalElements || 0,
        onPageChange: handlePageChange,
        page: params.page,
        onRowsPerPageChange: handleRowsPerPageChange,
        rowsPerPage: params.size,
        sortBy: params.orderBy || 'id',
        sortDirection: params.orderDir || 'desc',
        filteredData: fuelTypesPage,
        handleStartExportFuelTypes: handleStartExportFuelTypes,
        onResetFilters: onResetFilters,
        keyword: keyword,
        setKeyword: setKeyword,
    };
}