"use client"

import React from "react"
import { TableCell, IconButton, Switch, Chip, Typography, Box } from "@mui/material"
import { TableColumn } from "@/shared/types/common"
import { TrajetStyledRow } from "@/shared/sections/contract/styles"
import FontAwesome from "@/shared/components/fontawesome"
import { faEdit } from "@fortawesome/free-solid-svg-icons"
import { FuelType } from "@/shared/types/fleet-management/fuelType"
import { useFuelTypeStore } from "@/shared/api/stores/fleet-service/fuel-type-store"

type Props = {
  selected: boolean
  row: FuelType
  onSelectRow: VoidFunction
  onEdit: (row: FuelType) => void
  columns: TableColumn[]
}

export default function FuelTypeTableRow({
  row,
  selected,
  onSelectRow,
  onEdit,
  columns,
}: Props) {

  const { toggleFuelTypeActive } = useFuelTypeStore();

  const handleToggleActivation = async () => {
    await toggleFuelTypeActive(row.id, !row.active);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "id":
        return (
          <Typography variant="body2" sx={{ fontWeight: 500 }}>
            {value}
          </Typography>
        )

      case "name":
        return (
          <Typography variant="body2" sx={{ fontWeight: 500 }}>
            {value}
          </Typography>
        )

      case "active":
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        )

      case "actions":
        return (
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <IconButton
              size="small"
              onClick={() => onEdit(row)}
              sx={{
                color: "text.secondary",
                "&:hover": {
                  color: "primary.main",
                }
              }}
            >
              <FontAwesome icon={faEdit} />
            </IconButton>
            <Switch
              checked={row.active}
              onChange={() => handleToggleActivation()}
              size="small"
            />
          </Box>
        )

      default:
        return (
          <Typography variant="body2">
            {value}
          </Typography>
        )
    }
  }

  return (
    <TrajetStyledRow
      style={{ cursor: "pointer" }}
      sx={{
        "&:hover": {
          backgroundColor: "action.hover",
        }
      }}
    >
      {columns.map((column: TableColumn) => (
        <TableCell
          key={column.id}
          align={column.align}
          sx={{
            py: 1.5
          }}
        >
          {renderCellContent(column.id, row[String(column.id) as keyof FuelType])}
        </TableCell>
      ))}
    </TrajetStyledRow>
  )
}