import React from 'react';
import { Box, Stack } from '@mui/material';
import { ModeType, TableType } from '@/shared/types/common';
import CustomTabs from '@/shared/components/tabs/tabs-custom';
import TableManager from '@/shared/components/table/table-manager';
import { AddButton } from '@/shared/components/table/styles';
import FontAwesome from '@/shared/components/fontawesome';
import { faSquarePlus } from '@fortawesome/free-solid-svg-icons';
import { UnsavedChangesDialog } from '@/shared/components/dialog/UnsavedChangesDialog';
import { useAssignmentHistoryTabs } from '../../hooks/use-assignment-history-tabs';
import { useAssignmentHistoryTable } from '../../hooks/use-assignment-history-table';
import AssignmentHistoryForm from './assignment-history-form';
import HistoryAssignmentRow from './history-assignment-row';
import { AssignmentHistory } from '@/shared/types/flottes/vehicle-affectation';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

export default function AssignmentHistoryListView({mainMode}: {mainMode: ModeType}) {
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    setTabHasUnsavedChanges,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
  } = useAssignmentHistoryTabs();

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleSave,
  } = useAssignmentHistoryTable();

  const activeTabData = React.useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab);
  }, [tabs, activeTab]);
  const displayedData = mainMode === 'add' ? [] : filteredData;
  const renderRow = (row: AssignmentHistory) => (
    <HistoryAssignmentRow
      key={row.id}
      row={row}
      dense={table.dense}
      columns={tableHead}
      handleDetails={handleView}
    />
  );

  return (
    <Box sx={{ display: 'flex', flexDirection: 'column', mt: 2 }}>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <AddButton
          onClick={handleTabAdd}
          variant="contained"
          disabled={mainMode === 'view'}
          endIcon={<FontAwesome icon={faSquarePlus} width={18} />}
        >
          Ajouter une attribution
        </AddButton>
      </Stack>
      <CustomTabs
        type={TableType.Chauffeur}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />
      <ConditionalComponent isValid={activeTab === 'list'}
        defaultComponent={
          <AssignmentHistoryForm
            key={activeTab}
            assignment={activeTabData?.content as AssignmentHistory}
            mode={activeTabData?.mode || 'view'}
            onSave={handleSave}
            onClose={() => handleCancel(activeTab)}
            onEdit={handleEdit}
            setHasUnsavedChanges={(value) => setTabHasUnsavedChanges(activeTab, value)}
            updateTabContent={updateTabContent}
            tabId={activeTab}
          />
        }
      >
        <TableManager
          filteredData={displayedData}
          table={table}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
          sx={{ backgroundColor: 'primary.lighter' }}
        />
      </ConditionalComponent>
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={() => handleConfirmDialogAction()}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </Box>
  );
} 