'use client';
import * as React from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  FormGroup,
  FormControlLabel,
  Checkbox,
  Typography,
  Divider,
} from '@mui/material';
import { ConfigKey } from '@/shared/_mock/flottes/_anomaly-config';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

const LABELS: Record<ConfigKey, string> = {
  multipleUsePerDay: 'Utilisation multiple par jour',
  fuelOutsideServiceArea: 'Plein hors zone de service',
  fuelOutsideServiceHours: 'Plein hors horaires de service',
  abnormallyHighVolume: 'Volume anormalement élevé',
  cardUsedDuringStop: 'Carte utilisée pendant arrêt',
  useWithoutPlannedTrip: 'Utilisation sans trajet prévu',
  repeatedStationShortInterval: 'Station répétée à court intervalle',
  abnormalConsumption: 'Consommation anormale',
  useOutsideTrip: 'Utilisation hors trajet',
  km_deviation: 'Écart de consommation anormale',
  avg_per_km: 'Moyenne consommation par km',
};

type Props = {
  open: boolean;
  onClose: () => void;
  sourceKey: ConfigKey | null;
  onApplySelectedKeys?: (keys: ConfigKey[]) => void;
};

export default function ApplyToOthersDialog({
  open,
  onClose,
  sourceKey,
  onApplySelectedKeys,
}: Props) {
  const [selected, setSelected] = React.useState<ConfigKey[]>([]);

  React.useEffect(() => {
    if (!open) setSelected([]);
  }, [open]);

  const ALL: ConfigKey[] = Object.keys(LABELS) as ConfigKey[];

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="sm"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 2,
          boxShadow: 8,
          overflow: 'hidden',
        },
      }}
    >
      <DialogTitle
        sx={{
          bgcolor: 'primary.main',
          color: 'primary.contrastText',
          fontWeight: 600,
          py: 1.5,
          px: 2.5,
          height: 70,
          display: 'flex',
          alignItems: 'center',
        }}
      >
        Appliquer cette configuration à d’autres anomalies
      </DialogTitle>

      <DialogContent dividers sx={{ px: 3, py: 2 }}>
        <ConditionalComponent
          isValid={!!sourceKey}
          defaultComponent={
            <Typography variant="body2" color="text.disabled" gutterBottom>
              Aucune source sélectionnée
            </Typography>
          }
        >
          <Typography variant="body2" gutterBottom>
            Source : <b>{LABELS[sourceKey as ConfigKey]}</b>
          </Typography>
        </ConditionalComponent>
        <Divider sx={{ mb: 2 }} />
        <Typography variant="subtitle2" gutterBottom>
          Choisissez les cibles :
        </Typography>
        <FormGroup>
          {ALL.filter((k) => k !== sourceKey).map((k) => (
            <FormControlLabel
              key={k}
              control={
                <Checkbox
                  checked={selected.includes(k)}
                  onChange={(e) =>
                    setSelected((cur) =>
                      e.target.checked ? [...cur, k] : cur.filter((x) => x !== k)
                    )
                  }
                />
              }
              label={LABELS[k]}
            />
          ))}
        </FormGroup>
      </DialogContent>

      <DialogActions sx={{ px: 3, pb: 2 }}>
        <Button onClick={onClose}>Annuler</Button>
        <Button
          variant="contained"
          onClick={() => {
            onApplySelectedKeys?.(selected);
            onClose();
          }}
          disabled={selected.length === 0}
        >
          Appliquer
        </Button>
      </DialogActions>
    </Dialog>
  );
}
