'use client';
import * as React from 'react';
import dayjs, { Dayjs } from 'dayjs';
import { Box, Stack, Typography, Grid } from '@mui/material';
import CalendarMonthOutlinedIcon from '@mui/icons-material/CalendarMonthOutlined';

import { faClock } from '@fortawesome/free-regular-svg-icons';

import DaySelector from './day-selector';
import { RoadMapStyledDatePicker, RoadMapStyledTimePicker } from '@/shared/theme/css';
import { CustomDateIcon } from '@/shared/components/form/form-field';
import FontAwesome from '@/shared/components/fontawesome';
import { LocalizationProvider } from '@mui/x-date-pickers';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';
import { fr } from 'date-fns/locale';

export type ScheduleDetailsValue = {
  startDate: Dayjs | Date | null;
  endDate: Dayjs | Date | null;
  days: number[];
  startHour: Dayjs | Date | null;
  endHour: Dayjs | Date | null;
};

type Props = {
  title?: string;
  value: ScheduleDetailsValue;
  onChange: (patch: Partial<ScheduleDetailsValue>) => void;
  disabled?: boolean;
};

export default function ScheduleDetailsEditor({
  title = 'Calendrier & Horaires',
  value,
  onChange,
  disabled,
}: Props) {
  const { startDate, endDate, days, startHour, endHour } = value;

  return (
    <Stack spacing={1}>
      <Stack direction="row" alignItems="center" gap={1}>
        <CalendarMonthOutlinedIcon fontSize="small" />
        <Typography variant="subtitle2">{title}</Typography>
      </Stack>

      <Box
        sx={{
          border: '1px dashed',
          borderColor: 'divider',
          p: 2,
          borderRadius: 1,
          bgcolor: 'background.paper',
        }}
      >
        <Grid container spacing={2}>
          <Grid item xs={12} md={6}>
            <RoadMapStyledDatePicker
              value={startDate}
              onChange={(val) => onChange({ startDate: val ?? null })}
              views={['year', 'month', 'day']}
              slots={{ openPickerIcon: CustomDateIcon }}
              slotProps={{
                textField: {
                  placeholder: 'Démarrage le',
                  InputLabelProps: { shrink: true },
                  fullWidth: true,
                },
              }}
              sx={{ width: '100%' }}
            />
          </Grid>

          <Grid item xs={12} md={6}>
            <RoadMapStyledDatePicker
              value={endDate}
              onChange={(val) => onChange({ endDate: val ?? null })}
              views={['year', 'month', 'day']}
              slots={{ openPickerIcon: CustomDateIcon }}
              slotProps={{
                textField: {
                  placeholder: 'Fin le',
                  InputLabelProps: { shrink: true },
                  fullWidth: true,
                },
              }}
              sx={{ width: '100%' }}
            />
          </Grid>

          <Grid item xs={12}>
            <Typography variant="body2" sx={{ mb: 0.5 }}>
              Jours :
            </Typography>
            <DaySelector
              value={days ?? []}
              onChange={(v) => onChange({ days: v })}
              disabled={disabled}
            />
          </Grid>

          <Grid item xs={12} md={6}>
            <RoadMapStyledTimePicker
              label="Heure début :"
              value={startHour}
              onChange={(val) => onChange({ startHour: val ?? null })}
              views={['hours', 'minutes']}
              format="HH:mm"
              slots={{
                openPickerIcon: () => <FontAwesome icon={faClock} />,
              }}
              slotProps={{
                textField: {
                  placeholder: "Sélectionner l'heure",
                  fullWidth: true,
                  InputLabelProps: { shrink: true },
                },
              }}
            />
          </Grid>

          <Grid item xs={12} md={6}>
            <RoadMapStyledTimePicker
              label="Heure fin :"
              value={endHour}
              onChange={(val) => onChange({ endHour: val ?? null })}
              views={['hours', 'minutes']}
              format="HH:mm"
              slots={{
                openPickerIcon: () => <FontAwesome icon={faClock} />,
              }}
              slotProps={{
                textField: {
                  placeholder: "Sélectionner l'heure",
                  fullWidth: true,
                  InputLabelProps: { shrink: true },
                },
              }}
            />
          </Grid>
        </Grid>
      </Box>
    </Stack>
  );
}
