import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Stack,
  Typography,
} from '@mui/material';
import { useState } from 'react';
import { StyledTextField } from '@/shared/theme/css';
import { pxToRem } from '@/shared/theme/typography';
import { outlinedColor } from '@/shared/components/table/styles';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faFolder } from '@fortawesome/free-solid-svg-icons';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface AddFolderDialogProps {
  open: boolean;
  onClose: () => void;
  onAdd: (folderName: string) => void;
  currentPath: string[];
}

export default function AddFolderDialog({ 
  open, 
  onClose, 
  onAdd, 
  currentPath 
}: AddFolderDialogProps) {
  const [folderName, setFolderName] = useState('');
  const [error, setError] = useState('');

  const handleSubmit = (event: React.FormEvent) => {
    event.preventDefault();
    
    if (!folderName.trim()) {
      setError('Le nom du dossier est obligatoire');
      return;
    }
    
    if (folderName.includes('/') || folderName.includes('\\')) {
      setError('Le nom du dossier ne peut pas contenir / ou \\');
      return;
    }
    
    onAdd(folderName.trim());
    handleClose();
  };

  const handleClose = () => {
    setFolderName('');
    setError('');
    onClose();
  };

  return (
    <Dialog open={open} onClose={handleClose} maxWidth="sm" fullWidth>
      <DialogTitle
        sx={{
          fontSize: pxToRem(20),
          color: 'primary.main',
          fontWeight: (theme) => theme.typography.fontWeightBold,
          display: 'flex',
          alignItems: 'center',
          gap: 1,
        }}
      >
        <FontAwesomeIcon icon={faFolder} />
        Nouveau dossier
      </DialogTitle>
      
      <form onSubmit={handleSubmit}>
        <DialogContent>
          <Stack spacing={3}>
            <ConditionalComponent isValid={currentPath.length > 0}>
              <Typography variant="body2" color="text.secondary">
                Créer un nouveau dossier dans : {currentPath.join(' > ')}
              </Typography>
            </ConditionalComponent>
            
            <StyledTextField
              fullWidth
              label="Nom du dossier"
              placeholder="Entrez le nom du dossier"
              value={folderName}
              onChange={(e) => {
                setFolderName(e.target.value);
                if (error) setError('');
              }}
              error={!!error}
              helperText={error}
              autoFocus
              required
            />

            <ConditionalComponent isValid={!error && folderName.length > 0}>
              <Typography variant="caption" color="success.main">
                Le dossier &quot;{folderName}&quot; sera créé
              </Typography>
            </ConditionalComponent>
          </Stack>
        </DialogContent>
        
        <DialogActions>
          <Button
            sx={{ ...outlinedColor }}
            variant="outlined"
            onClick={handleClose}
          >
            Annuler
          </Button>
          <Button type="submit" variant="contained">
            Créer le dossier
          </Button>
        </DialogActions>
      </form>
    </Dialog>
  );
} 