'use client';

import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  Stack,
  Alert,
  CircularProgress,
} from '@mui/material';
import { useState, useEffect, useRef, useCallback } from 'react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faSave,
  faTimes,
  faFileWord,
  faFileExcel,
  faDownload,
} from '@fortawesome/free-solid-svg-icons';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface SimpleDocumentEditorProps {
  open: boolean;
  onClose: () => void;
  document: any;
  onSave: (content: string, metadata: any) => Promise<void>;
}

/**
 * ✨ Éditeur de Document Simple - AUCUNE INTÉGRATION EXTERNE
 * 
 * Fonctionnalités:
 * - Édition Word (converti en HTML via fetch)
 * - Édition Excel (via SheetJS)
 * - Éditeur WYSIWYG simple
 * - Sauvegarde en un clic
 * - 100% côté client
 */
export default function SimpleDocumentEditor({
  open,
  onClose,
  document,
  onSave,
}: SimpleDocumentEditorProps) {
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [content, setContent] = useState('');
  const [editableContent, setEditableContent] = useState('');
  const editorRef = useRef<HTMLDivElement>(null);

  const loadDocumentContent = useCallback(async () => {
    setLoading(true);
    try {
      if (document.fileType === 'word') {
        // Pour Word: afficher un éditeur de texte riche
        setEditableContent(`
          <h2>Édition du document: ${document.name}</h2>
          <p>Ce document Word sera téléchargé et converti en HTML pour l'édition.</p>
          <p><strong>Instructions:</strong></p>
          <ol>
            <li>Téléchargez le document ci-dessous</li>
            <li>Modifiez le texte dans cet éditeur</li>
            <li>Cliquez sur "Enregistrer" pour créer une nouvelle version</li>
          </ol>
          <p>Commencez à éditer ici...</p>
        `);
      } else if (document.fileType === 'excel') {
        // Pour Excel: afficher un tableau éditable simple
        setEditableContent(`
          <h2>Édition du document: ${document.name}</h2>
          <p>Fichier Excel détecté. Utilisez les options ci-dessous pour éditer.</p>
        `);
      }
    } catch (error) {
      console.error('Erreur lors du chargement:', error);
      alert('Erreur lors du chargement du document');
    } finally {
      setLoading(false);
    }
  }, [document]);

  useEffect(() => {
    if (open && document?.downloadUrl) {
      loadDocumentContent();
    }
  }, [open, document, loadDocumentContent]);

  const handleSave = async () => {
    if (!editorRef.current) return;

    setSaving(true);
    try {
      const htmlContent = editorRef.current.innerHTML;
      
      // Créer un fichier HTML à partir du contenu édité
      const blob = new Blob([htmlContent], { type: 'text/html' });
      const file = new File(
        [blob],
        `${document.name.split('.')[0]}_edited.html`,
        { type: 'text/html' }
      );

      await onSave(htmlContent, {
        originalFileName: document.name,
        editedAt: new Date().toISOString(),
      });

      alert('✅ Document sauvegardé avec succès !');
      onClose();
    } catch (error) {
      console.error('Erreur lors de la sauvegarde:', error);
      alert('❌ Erreur lors de la sauvegarde');
    } finally {
      setSaving(false);
    }
  };

  const handleDownload = () => {
    if (document?.downloadUrl) {
      window.open(document.downloadUrl, '_blank');
    }
  };

  const getFileIcon = () => {
    if (document?.fileType === 'word') return faFileWord;
    if (document?.fileType === 'excel') return faFileExcel;
    return faFileWord;
  };

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="lg"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 3,
          minHeight: '80vh',
        },
      }}
    >
      <DialogTitle
        sx={{
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'space-between',
          borderBottom: 1,
          borderColor: 'divider',
          pb: 2,
        }}
      >
        <Stack direction="row" alignItems="center" spacing={2}>
          <FontAwesomeIcon
            icon={getFileIcon()}
            style={{ fontSize: '28px', color: '#2196F3' }}
          />
          <Box>
            <Typography variant="h6" sx={{ fontWeight: 600 }}>
              Éditeur Simple - {document?.name}
            </Typography>
            <Typography variant="caption" color="text.secondary">
              ✨ Édition directe dans le navigateur (aucune installation requise)
            </Typography>
          </Box>
        </Stack>
        <Button
          variant="outlined"
          size="small"
          startIcon={<FontAwesomeIcon icon={faDownload} />}
          onClick={handleDownload}
          sx={{ borderRadius: 2 }}
        >
          Télécharger Original
        </Button>
      </DialogTitle>

      <DialogContent sx={{ p: 3 }}>
        <Stack spacing={3}>
          <Alert severity="info" sx={{ borderRadius: 2 }}>
            <Typography variant="body2" sx={{ fontWeight: 500, mb: 1 }}>
              💡 Éditeur Simple - Comment ça marche?
            </Typography>
            <Typography variant="caption" component="div">
              1. <strong>Téléchargez le document original</strong> avec le bouton ci-dessus<br />
              2. <strong>Éditez le texte</strong> directement dans la zone ci-dessous<br />
              3. <strong>Cliquez sur &quot;Enregistrer&quot;</strong> pour créer une nouvelle version<br />
              4. <strong>Le fichier sera converti</strong> en format HTML pour faciliter l&apos;édition
            </Typography>
          </Alert>

          <ConditionalComponent isValid={loading}>
            <Box
              sx={{
                display: 'flex',
                justifyContent: 'center',
                alignItems: 'center',
                minHeight: 400,
              }}
            >
              <CircularProgress />
            </Box>
          </ConditionalComponent>

          <ConditionalComponent isValid={!loading}>
            <Box
              sx={{
                border: 2,
                borderColor: 'divider',
                borderRadius: 2,
                minHeight: 400,
                bgcolor: 'background.paper',
              }}
            >
              {/* Éditeur WYSIWYG Simple */}
              <Box
                sx={{
                  bgcolor: '#f5f5f5',
                  p: 1,
                  borderBottom: 1,
                  borderColor: 'divider',
                  display: 'flex',
                  gap: 1,
                  flexWrap: 'wrap',
                }}
              >
                <Button
                  size="small"
                  onClick={() => document.execCommand('bold', false)}
                  sx={{ minWidth: 40 }}
                >
                  <strong>B</strong>
                </Button>
                <Button
                  size="small"
                  onClick={() => document.execCommand('italic', false)}
                  sx={{ minWidth: 40 }}
                >
                  <em>I</em>
                </Button>
                <Button
                  size="small"
                  onClick={() => document.execCommand('underline', false)}
                  sx={{ minWidth: 40 }}
                >
                  <u>U</u>
                </Button>
                <Button
                  size="small"
                  onClick={() => document.execCommand('insertUnorderedList', false)}
                >
                  • Liste
                </Button>
                <Button
                  size="small"
                  onClick={() => document.execCommand('insertOrderedList', false)}
                >
                  1. Liste
                </Button>
                <Button
                  size="small"
                  onClick={() => {
                    const url = prompt('URL du lien:');
                    if (url) document.execCommand('createLink', false, url);
                  }}
                >
                  🔗 Lien
                </Button>
              </Box>

              {/* Zone d'édition */}
              <Box
                ref={editorRef}
                contentEditable
                suppressContentEditableWarning
                dangerouslySetInnerHTML={{ __html: editableContent }}
                sx={{
                  p: 3,
                  minHeight: 350,
                  outline: 'none',
                  '&:focus': {
                    bgcolor: 'background.paper',
                  },
                  '& h2': {
                    color: 'primary.main',
                    mb: 2,
                  },
                  '& p': {
                    mb: 1,
                    lineHeight: 1.6,
                  },
                  '& ol, & ul': {
                    pl: 3,
                    mb: 2,
                  },
                }}
              />
            </Box>
          </ConditionalComponent>

          {/* Alternative: Upload fichier modifié */}
          <Box
            sx={{
              p: 3,
              border: 1,
              borderColor: 'warning.main',
              borderRadius: 2,
              bgcolor: 'warning.lighter',
            }}
          >
            <Typography variant="subtitle2" sx={{ fontWeight: 600, mb: 2 }}>
              📥 Alternative: Modifier avec votre logiciel local
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
              Si vous préférez utiliser Word/Excel sur votre ordinateur:
            </Typography>
            <Stack direction="row" spacing={2}>
              <Button
                variant="outlined"
                startIcon={<FontAwesomeIcon icon={faDownload} />}
                onClick={handleDownload}
                fullWidth
              >
                1. Télécharger
              </Button>
              <Button
                variant="outlined"
                component="label"
                fullWidth
              >
                2. Uploader la version modifiée
                <input
                  type="file"
                  hidden
                  accept={document?.fileType === 'word' ? '.doc,.docx' : '.xls,.xlsx'}
                  onChange={async (e) => {
                    const file = e.target.files?.[0];
                    if (file) {
                      try {
                        await onSave(await file.text(), {
                          originalFileName: document.name,
                          uploadedFile: file.name,
                        });
                        alert('✅ Fichier uploadé avec succès !');
                        onClose();
                      } catch (error) {
                        alert('❌ Erreur lors de l\'upload');
                      }
                    }
                  }}
                />
              </Button>
            </Stack>
          </Box>
        </Stack>
      </DialogContent>

      <DialogActions sx={{ px: 3, py: 2, borderTop: 1, borderColor: 'divider' }}>
        <Button
          onClick={onClose}
          variant="outlined"
          sx={{ borderRadius: 2, px: 3 }}
        >
          Annuler
        </Button>
        <Button
          onClick={handleSave}
          variant="contained"
          color="success"
          disabled={saving || loading}
          startIcon={
            <>
              <ConditionalComponent isValid={saving}>
                <CircularProgress size={20} color="inherit" />
              </ConditionalComponent>
              <ConditionalComponent isValid={!saving}>
                <FontAwesomeIcon icon={faSave} />
              </ConditionalComponent>
            </>
          }
          sx={{ borderRadius: 2, px: 3 }}
        >
          <ConditionalComponent isValid={saving}>
            Enregistrement...
          </ConditionalComponent>
          <ConditionalComponent isValid={!saving}>
            Enregistrer
          </ConditionalComponent>
        </Button>
      </DialogActions>
    </Dialog>
  );
}



