import React from 'react';
import TableCell from '@mui/material/TableCell';
import { IIntervention } from '@/shared/types/flottes/intervention';
import TableActionColumn from '@/shared/components/table/table-action-column';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { _OPTION_STATUT } from '@/shared/_mock/flottes/_intervention';
import { ContratStyledRow, getStatusColor, StatusChip } from '@/shared/sections/contract/styles';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';


type Props = {
  handleEditIntervention: (intervention: IIntervention) => void;
  handleDetailsIntervention: (intervention: IIntervention) => void;
  handleCopyIntervention?: (intervention: IIntervention) => void;
  selected: boolean;
  row: IIntervention;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IIntervention, newStatus: string, field: keyof IIntervention) => void;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier réparation',
  duplicate: 'Copier réparation',
  active: 'Réparation active',
  inactive: 'Réparation inactive',
};

export default function InterventionTableRow({
  row,
  handleEditIntervention,
  handleDetailsIntervention,
  handleCopyIntervention,

  columns,
  onStatusChange
}: Props) {

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateIntervention':
      case 'dateDebutImmobilisation':
      case 'dateFinImmobilisation':
        return formatDate(value, dateFormat.isoDate);
      case 'statut':
        return (
          <StatusMenu
            value={value}
            options={_OPTION_STATUT}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'statut' as keyof IIntervention)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{width:"180px"}}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
    {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsIntervention(row)}
            key={column.id}
            align={column.align || 'left'}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'statut' ? 0 : 'inherit', ...(index === 0 && {position: "relative"}), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getStatusColor(row.statut) }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IIntervention])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditIntervention(row)}
          onDuplicate={handleCopyIntervention ? () => handleCopyIntervention(row) : undefined}
          onToggle={(checked) => void 0}
          isActive={row.isActive}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}
