"use client"
import TableCell from "@mui/material/TableCell"
import TableActionColumn from "@/shared/components/table/table-action-column"
import SvgColor from "@/shared/components/svg-color"
import StatusMenu from "@/shared/components/table/StatusMenu"
import type { TableColumn } from "@/shared/types/common"
import { ContratStyledRow, getStatusColor, StatusChip } from "@/shared/sections/contract/styles"
import { RepairStatusOptions, 
  type VehicleRepairListResponse,
  type RepairStatus,
  VehicleRepair, } from "@/shared/types/fleet-management/vehicle-repair"
import { useVehicleRepairStore } from "@/shared/api/stores/fleet-service/vehicle-repair-store"
import { useInfrastructureStore } from "@/shared/api/stores/admin-service/infrastructureStore"
import { useEffect, useState } from "react"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { useVehicleStore } from "@/shared/api/stores/vehicleStore"
import { useServiceProviderStore } from "@/shared/api/stores/fleet-service/service-provider-store"

type Props = {
  handleEditRepair: (repair: VehicleRepairListResponse) => void
  handleDetailsRepair: (repair: VehicleRepairListResponse) => void
  handleCopyRepair: (repair: VehicleRepairListResponse) => void
  selected: boolean
  row: VehicleRepairListResponse
  onSelectRow: VoidFunction
  columns: TableColumn[]
  onStatusChange: (row: VehicleRepairListResponse, newStatus: string, field: keyof VehicleRepairListResponse) => void;
}

const TOOLTIP_TITLES = {
  edit: "Modifier réparation",
  duplicate: "Copier réparation",
  active: "Réparation active",
  inactive: "Réparation inactive",
}

export default function VehicleRepairTableRow({
  row,
  handleEditRepair,
  handleDetailsRepair,
  handleCopyRepair,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {
  const { toggleActive, setStatus } = useVehicleRepairStore()

  const handleToggleActivation = async () => {
    await toggleActive(row.id)
  }


  const handleToggleRepairStatus = async (newStatus: RepairStatus) => {
    await setStatus(row.id, newStatus);
    onStatusChange(row, newStatus, 'repairStatus' as keyof VehicleRepairListResponse)
  }
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "repairStatus":
        return (
          <StatusMenu
            value={value}
            options={RepairStatusOptions}
            onChange={(newStatus) => handleToggleRepairStatus(newStatus as RepairStatus)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{ width: "180px" }}
          />
        )
      case "costTTC":
        return value ? `${value.toFixed(2)} €` : "-"
      default:
        return value
    }
  }

  return (
    <ContratStyledRow hover style={{ cursor: "pointer" }}>
      {columns.map((column: any, index: number) => (
        <TableCell
          onDoubleClick={() => handleDetailsRepair(row)}
          key={column.id}
          align={column.align}
          sx={{
            whiteSpace: "nowrap",
            py: column.id === "repairStatus" ? 0 : "inherit",
            ...(index === 0 && { position: "relative" }),
          }}
        >
              <ConditionalComponent isValid={index === 0}>
                  <SvgColor
                      src="/assets/icons/ic_border.svg"
                      sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getStatusColor(row.repairStatus) }}
                  />
              </ConditionalComponent>
          {renderCellContent(column.id, row[column.id as keyof VehicleRepairListResponse])}
        </TableCell>
      ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: "nowrap" }}>
        <TableActionColumn
          handleEdit={() => handleEditRepair(row)}
          onDuplicate={() => handleCopyRepair(row)}
          onToggle={() => handleToggleActivation()}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  )
}
