import React from "react"
import { TableCell, Chip, Box, Typography } from "@mui/material"
import { TableColumn } from "@/shared/types/common"
import { dateFormat, formatDate } from "@/utils/format-time"
import { TrajetStyledRow } from "@/shared/sections/contract/styles"
import { ISuiviCarte, TypeCarburant, PetrolierType } from "@/shared/types/flottes/suiviCartes"
import FontAwesome from "@/shared/components/fontawesome"
import { faGasPump, faBolt, faDroplet } from "@fortawesome/free-solid-svg-icons"

type Props = {
  selected: boolean
  row: ISuiviCarte
  onSelectRow: VoidFunction
  columns: TableColumn[]
}

export default function SuiviCartesTableRow({
  row,
  selected,
  onSelectRow,
  columns,
}: Props) {
  const getCarburantIcon = (type: string) => {
    switch (type) {
      case TypeCarburant.Electrique:
        return faBolt
      case TypeCarburant.Diesel:
        return faDroplet
      case TypeCarburant.Essence:
        return faGasPump
      default:
        return faGasPump
    }
  }

  const getCarburantColor = (type: string) => {
    switch (type) {
      case TypeCarburant.Electrique:
        return "#4CAF50"
      case TypeCarburant.Diesel:
        return "#FF9800"
      case TypeCarburant.Essence:
        return "#f44336"
      case TypeCarburant.Hybride:
        return "#9C27B0"
      default:
        return "#757575"
    }
  }

  const getPetrolierColor = (petrolier: string) => {
    switch (petrolier) {
      case PetrolierType.DKV:
        return { bg: "#E3F2FD", color: "#1976D2" }
      case PetrolierType.TOTAL:
        return { bg: "#FFF3E0", color: "#F57C00" }
      case PetrolierType.SHELL:
        return { bg: "#FFEBEE", color: "#D32F2F" }
      case PetrolierType.BP:
        return { bg: "#E8F5E8", color: "#388E3C" }
      case PetrolierType.ESSO:
        return { bg: "#F3E5F5", color: "#7B1FA2" }
      default:
        return { bg: "#F5F5F5", color: "#757575" }
    }
  }

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "numeroCarteCarburant":
        return (
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <Typography variant="body2" sx={{ fontWeight: 600, color: "primary.main" }}>
              {value}
            </Typography>
          </Box>
        )

      case "nomConducteur":
        return (
          <Typography variant="body2" sx={{ fontWeight: 500 }}>
            {value}
          </Typography>
        )

      case "dateTransaction":
        return (
          <Typography variant="body2">
            {formatDate(value, dateFormat.isoDate)}
          </Typography>
        )

      case "petrolier":
        const petrolierColors = getPetrolierColor(value)
        return (
          <Chip
            label={value}
            size="small"
            sx={{
              backgroundColor: petrolierColors.bg,
              color: petrolierColors.color,
              fontWeight: 500,
              minWidth: "60px"
            }}
          />
        )

      case "typeCarburant":
        const carburantColor = getCarburantColor(value)
        return (
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <FontAwesome 
              icon={getCarburantIcon(value)} 
              sx={{ color: carburantColor, fontSize: "1rem" }}
            />
            <Typography variant="body2" sx={{ color: carburantColor, fontWeight: 500 }}>
              {value}
            </Typography>
          </Box>
        )

      case "montantTransaction":
        return (
          <Typography variant="body2" sx={{ fontWeight: 600, color: "success.main" }}>
            {typeof value === "number" ? `${value.toFixed(2)} €` : `${value} €`}
          </Typography>
        )

      case "quantite":
        return (
          <Typography variant="body2">
            {value} L
          </Typography>
        )

      case "adresseTransaction":
        return (
          <Typography 
            variant="body2" 
            sx={{ 
              maxWidth: "300px",
              overflow: "hidden",
              textOverflow: "ellipsis",
              whiteSpace: "nowrap"
            }}
            title={value}
          >
            {value}
          </Typography>
        )

      default:
        return value
    }
  }

  return (
    <TrajetStyledRow 
      style={{ cursor: "pointer" }}
      sx={{
        "&:hover": {
          backgroundColor: "primary.lighter",
        }
      }}
    >
      {columns.filter(c => c.id !== "").map((column: TableColumn) => (
        <TableCell
          key={column.id}
          align={column.align}
          sx={{ 
            whiteSpace: "nowrap",
            py: 2
          }}
        >
          {renderCellContent(column.id, row[column.id as keyof ISuiviCarte])}
        </TableCell>
      ))}
    </TrajetStyledRow>
  )
} 