import type React from 'react';
import { Grid, Typography } from '@mui/material';
import { TableType, type FormFieldType, type ModeType, type TableColumn } from '@/shared/types/common';
import ActionButtons from '@/shared/components/form/buttons-action';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import {
  mainIncidentFormFields,
  descriptionIncidentFormFields,
  suiviAdminIncidentFormFields,
} from '../utils/form-fields-vehicle-incident';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { VehicleIncidentRequest } from '@/shared/types/fleet-management/vehicle-incident';

interface InterventionTabProps {
  renderField: (field: FormFieldType<VehicleIncidentRequest>) => React.ReactNode;
  mode: ModeType;
  onSave: () => void;
  onClose: (isSaved: boolean) => void;
  onEdit: () => void;
  tableHead: TableColumn[];
  incident?: VehicleIncidentRequest;
  departmentOptions: { value: string; label: string }[],
  vehicleOptions: { value: string; label: string }[],
  driverOptions: { value: string; label: string }[],
  serviceProviderOptions: { value: string; label: string }[],
  interventionOptions: { value: string; label: string }[]
}

function SectionTitle({ children }: { children: React.ReactNode }) {
  return (
    <Typography
      variant="h6"
      sx={{
        color: 'primary.main',
        my: 3,
        fontWeight: 'bold',
      }}
    >
      {children}
    </Typography>
  );
}

export default function VehicleIncidentDetails({
  mode,
  onSave,
  onClose,
  onEdit,
  renderField,
  incident,
  tableHead,
  departmentOptions,
  vehicleOptions,
  driverOptions,
  serviceProviderOptions,
  interventionOptions
}: InterventionTabProps) {
  const isReadOnly = mode === 'view';

  return (
    <>
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons onEdit={onEdit} onExport={() => void 0}
          tooltipTitle={TableType.VehicleIncident}
          dataRow={incident}
          tableHead={tableHead} />
      </ConditionalComponent>
      <form>
        <Grid container spacing={5}>
          {mainIncidentFormFields(departmentOptions, vehicleOptions, driverOptions).map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
        </Grid>

        <SectionTitle>Description du sinistre</SectionTitle>
        <Grid container spacing={5}>
          {descriptionIncidentFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
        </Grid>

        <SectionTitle>Suivi administratif</SectionTitle>
        <Grid container spacing={5}>
          {suiviAdminIncidentFormFields(serviceProviderOptions,interventionOptions).map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}

          <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
            <ActionButtons
                onSave={onSave}
                onCancel={() => {
                  onClose(false);
                }}
                mode={mode}
              />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </>
  );
}