import React from 'react';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { ContratStyledRow, StatusChip } from '@/shared/sections/contract/styles';
import { _INCIDENT_STATUS_OPTIONS } from '@/shared/_mock/flottes/_Incidents';
import { IncidentStatus, IncidentStatusOptions, VehicleIncident } from '@/shared/types/fleet-management/vehicle-incident';
import { useVehicleIncidentStore } from '@/shared/api/stores/fleet-service/vehicle-incident-store';
import { Chip } from '@mui/material';

type Props = {
  handleEditIncident: (incident: VehicleIncident) => void;
  handleDetailsIncident: (incident: VehicleIncident) => void;
  row: VehicleIncident;
  columns: TableColumn[];
  onStatusChange: (row: VehicleIncident, newStatus: string, field: keyof VehicleIncident) => void;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier sinistre',
  active: 'Sinistre actif',
  inactive: 'Sinistre inactif',
};

export default function VehicleIncidentTableRow({
  row,
  handleEditIncident,
  handleDetailsIncident,
  columns
}: Props) {

  const {
    toggleActive,
    toggleStatus
  } = useVehicleIncidentStore();

  const handleToggleVehicleIncidentStatus = async (newStatus: IncidentStatus) => {
    await toggleStatus(row.id, newStatus);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'incidentDate':
        return formatDate(value, dateFormat.isoDate);
      case 'serviceProviderName':
        return row.serviceProvider?.name;
      case 'interventionName':
        return row.serviceProvider?.intervention.name;
      case 'status':
        return (
          <StatusMenu
            value={value}
            options={IncidentStatusOptions}
            onChange={(newStatus) => handleToggleVehicleIncidentStatus(newStatus as IncidentStatus)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{ width: "150px" }}
          />
        );
      case "active":
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        )
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsIncident(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : 'inherit', }}
          >
            {renderCellContent(column.id, row[column.id as keyof VehicleIncident])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditIncident(row)}
          onToggle={(checked) => toggleActive(row.id, checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}