import React from 'react';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import SvgColor from '@/shared/components/svg-color';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { TableColumn } from '@/shared/types/common';
import { ContratStyledRow, StatusChip } from '@/shared/sections/contract/styles';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { getVehicleStatusColor, Vehicle, VehicleListResponse, VehicleStatus, VehicleStatusOptions } from '@/shared/types/fleet-management/vehicle';
import { useVehicleStore } from '@/shared/api/stores/fleet-service/vehicle-store';
import { dateFormat, formatDate } from '@/utils/format-time';
import { Chip } from '@mui/material';

type Props = {
  handleEditVehicle: (vehicle: VehicleListResponse) => void;
  handleDetailsVehicle: (vehicle: VehicleListResponse) => void;
  handleCopyVehicle: (vehicle: VehicleListResponse) => void;
  selected: boolean;
  row: VehicleListResponse;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
};

const TOOLTIP_TITLES = {
  edit: 'Modifier véhicule',
  duplicate: 'Copier véhicule',
  active: 'Véhicule actif',
  inactive: 'Véhicule inactif',
};

export default function VehicleTableRow({
  row,
  handleEditVehicle,
  handleDetailsVehicle,
  handleCopyVehicle,
  selected,
  onSelectRow,
  columns,
}: Props) {

  const { toggleActiveStatus, toggleVehicleStatus } = useVehicleStore();

  const handleToggleActivation = async (checked: boolean) => {
    await toggleActiveStatus(row.id, checked);
  };

  const handleToggleVehicleStatus = async (newStatus: VehicleStatus) => {
    await toggleVehicleStatus(row.id, newStatus);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'registrationDate':
        return formatDate(value, dateFormat.isoDate);
      case 'status':
        return (
          <StatusMenu
            value={value}
            options={VehicleStatusOptions}
            onChange={(newStatus) => handleToggleVehicleStatus(newStatus as VehicleStatus)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{ width: "180px" }}
          />
        );
      case "active":
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        )
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsVehicle(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : 'inherit', ...(index === 0 && { position: "relative" }), }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getVehicleStatusColor(row.status) }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof VehicleListResponse])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditVehicle(row)}
          onDuplicate={() => handleCopyVehicle(row)}
          onToggle={(checked) => handleToggleActivation(checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}