import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import { useVehicleBrandModelStore } from "@/shared/api/stores/fleet-service/vehicle-brand-model-store"
import { TableColumn } from "@/shared/types/common"
import { VehicleBrandModel, VehicleBrandModelPageFilterParams } from "@/shared/types/fleet-management/vehicle-brand-model"
import { useEffect, useState } from "react"
import { useSnackbar } from "notistack"

export const DEFAULT_VEHICLE_BRAND_MODEL_TABLE_HEAD: TableColumn[] = [
  { id: "id", label: "N°", type: "text" },
  { id: "brandName", label: "Marque", type: "text" },
  { id: "modelName", label: "Modèle", type: "text" },
  { id: "generalMaintenanceDate", label: "Date d'entretien", type: "date" },
  { id: "active", label: "Actif", align: "center" },
  { id: "", label: "Actions" },
]

const vehicleBrandModelConfig: TableConfig<VehicleBrandModel> = {
  initialData: [],
  defaultTableHead: DEFAULT_VEHICLE_BRAND_MODEL_TABLE_HEAD,
}

export function useVehicleBrandModelTable() {

  const { enqueueSnackbar } = useSnackbar();

  const [params, setParams] = useState<VehicleBrandModelPageFilterParams>({
    page: 0,
    size: 20,
    orderBy: 'id',
    orderDir: 'desc'
  });

  const {
    loading,
    error,
    vehicleBrandModelsPage,
    totalElements,

    fetchVehicleBrandModelsPage,
    startExportFiltredVehicleBrandModels
  } = useVehicleBrandModelStore();

  useEffect(() => {
    fetchVehicleBrandModelsPage(params);
  }, [params])

  const tableManager = useTableManager<any>({
    ...vehicleBrandModelConfig,
    initialData: vehicleBrandModelsPage || [],
  });

  const onResetFilters = () => {
    tableManager.handleResetFilters();
    setParams({
      page: 0,
      size: 20,
      orderBy: 'id',
      orderDir: 'desc'
    });
  };

  useEffect(() => {
    if (tableManager.table.rowsPerPage > 0) {
      setParams(prev => ({
        ...prev,
        page: tableManager.table.page,
        size: tableManager.table.rowsPerPage,
      }));
    }
  }, [tableManager.table.page, tableManager.table.rowsPerPage]);

  const isValidDateInput = (value: any): boolean => {
    if (!value || typeof value !== 'string') return false;
    const date = new Date(value);
    return date instanceof Date && !isNaN(date.getTime());
  };

  const isValidFilterValue = (value: any): boolean => {
    return value !== null && value !== undefined && value !== '';
  };

  useEffect(() => {
    const baseParams: VehicleBrandModelPageFilterParams = {
      page: tableManager.table.page || 0,
      size: tableManager.table.rowsPerPage || 20,
      orderBy: tableManager.table.orderBy || 'id',
      orderDir: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
    };

    const filterParams: Record<string, string | number> = {};

    Object.entries(tableManager.filters).forEach(([key, { value }]) => {
      if (key === 'id' && isValidFilterValue(value)) {
        filterParams[key] = Number(value);
      } else if (['generalMaintenanceDate'].includes(key) && isValidDateInput(value)) {
        const date = new Date(value);
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        filterParams[key] = `${year}-${month}-${day}T00:00:00`;
      } else if (isValidFilterValue(value)) {
        filterParams[key] = value;
      }
    });

    const newParams: VehicleBrandModelPageFilterParams = {
      ...baseParams,
      ...filterParams
    };

    const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
    if (paramsChanged) {
      setParams(newParams);
    }
  }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order]);

  const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
    setParams(prev => ({
      ...prev,
      page: newPage
    }));
  };

  const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const size = parseInt(event.target.value, 10);
    if (!isNaN(size) && size > 0) {
      setParams(prev => ({
        ...prev,
        size,
        page: 0
      }));
    }
  };

  const userId = localStorage.getItem('userId');

  const handleStartExportVehicleBrandModels = () => {
    if (userId) {
      startExportFiltredVehicleBrandModels(userId, params);
    } else {
      enqueueSnackbar("Nous n'avons pas pu récupérer l'identifiant de l'utilisateur connecté.");
    }
  };

  return {
    ...tableManager,
    loading,
    error,
    totalVehicleBrandModels: totalElements || 0,
    onPageChange: handlePageChange,
    page: params.page,
    onRowsPerPageChange: handleRowsPerPageChange,
    rowsPerPage: params.size,
    sortBy: params.orderBy || 'id',
    sortDirection: params.orderDir || 'desc',
    filteredData: vehicleBrandModelsPage,
    handleStartExportVehicleBrandModels: handleStartExportVehicleBrandModels,
    onResetFilters: onResetFilters,
  };
}