"use client"

import TableCell from "@mui/material/TableCell"
import TableActionColumn from "@/shared/components/table/table-action-column"
import SvgColor from "@/shared/components/svg-color"
import type { TableColumn } from "@/shared/types/common"
import { ContratStyledRow, getStatusColor } from "@/shared/sections/contract/styles"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { VehicleBrandModel, VehicleBrandModelRequest } from "@/shared/types/fleet-management/vehicle-brand-model"
import { useVehicleBrandModelStore } from "@/shared/api/stores/fleet-service/vehicle-brand-model-store"
import { dateFormat, formatDate } from "@/utils/format-time"
import { Box, Chip, IconButton, Switch } from "@mui/material"
import FontAwesome from "@/shared/components/fontawesome"
import { faEdit } from "@fortawesome/free-solid-svg-icons"

type Props = {
  handleEdit: (vehicleBrandModel: VehicleBrandModel) => void
  handleDetails: (vehicleBrandModel: VehicleBrandModel) => void
  selected: boolean
  row: VehicleBrandModel
  onSelectRow: VoidFunction
  columns: TableColumn[]
}

const TOOLTIP_TITLES = {
  edit: "Modifier marque / modèle",
  duplicate: "Copier marque / modèle",
  active: "Marque / modèle actif",
  inactive: "Marque / modèle inactif",
}

export default function VehicleBrandModelTableRow({
  row,
  handleEdit,
  handleDetails,
  selected,
  onSelectRow,
  columns,
}: Props) {

  const { toggleActiveStatus } = useVehicleBrandModelStore();

  const handleToggleActivation = async () => {
    await toggleActiveStatus(row.id, !row.active);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'generalMaintenanceDate':
        return formatDate(value, dateFormat.isoDate);
      case "active":
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        )
      case "actions":
        return (
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <IconButton
              size="small"
              onClick={() => handleEdit(row)}
              sx={{
                color: "text.secondary",
                "&:hover": {
                  color: "primary.main",
                }
              }}
            >
              <FontAwesome icon={faEdit} />
            </IconButton>
            <Switch
              checked={row.active}
              onChange={() => handleToggleActivation()}
              size="small"
            />
          </Box>
        )
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: "pointer" }}>

      {columns
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: "nowrap", py: column.id === "active" ? 0 : "inherit", ...(index === 0 && { position: "relative" }), }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getStatusColor(row.id) }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof VehicleBrandModel])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: "nowrap" }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onToggle={(checked) => toggleActiveStatus(row.id, checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  )
}