import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import { useVehicleTypeStore } from "@/shared/api/stores/fleet-service/vehicle-type-store"
import { TableColumn } from "@/shared/types/common"
import { VehicleType, VehicleTypePageFilterParams } from "@/shared/types/fleet-management/vehicleType"
import { useEffect, useState } from "react"
import { useSnackbar } from "notistack";


export const DEFAULT_VEHICLE_TYPE_TABLE_HEAD: TableColumn[] = [
  { id: "id", label: "N°", type: "text" },
  { id: "name", label: "Nom du type", type: "text" },
  { id: "active", label: "Actif", type: "boolean", align: "center" },
  { id: "actions", label: "Actions" },
]

const vehicleTypeConfig: TableConfig<VehicleType> = {
  initialData: [],
  defaultTableHead: DEFAULT_VEHICLE_TYPE_TABLE_HEAD,
}

export function useVehicleTypeTable() {
  const { enqueueSnackbar } = useSnackbar();

  const [params, setParams] = useState<VehicleTypePageFilterParams>({
    page: 0,
    size: 20,
    orderBy: 'id',
    orderDir: 'desc'
  });

  const {
    loading,
    error,
    vehicleTypesPage,
    totalElements,
    fetchVehicleTypesPage,
    startExportFiltredVehicleTypes
  } = useVehicleTypeStore();

  useEffect(() => {
    fetchVehicleTypesPage(params);
  }, [params]);

  const tableManager = useTableManager<any>({
    ...vehicleTypeConfig,
    initialData: vehicleTypesPage || [],
  });

  const onResetFilters = () => {
    tableManager.handleResetFilters();
    setParams({
      page: 0,
      size: 20,
      orderBy: 'id',
      orderDir: 'desc'
    });
  };

  useEffect(() => {
    if (tableManager.table.rowsPerPage > 0) {
      setParams(prev => ({
        ...prev,
        page: tableManager.table.page,
        size: tableManager.table.rowsPerPage,
      }));
    }
  }, [tableManager.table.page, tableManager.table.rowsPerPage]);

  const isValidFilterValue = (value: any): boolean => {
    return value !== null && value !== undefined && value !== '';
  };

  useEffect(() => {
    const baseParams: VehicleTypePageFilterParams = {
      page: tableManager.table.page || 0,
      size: tableManager.table.rowsPerPage || 20,
      orderBy: tableManager.table.orderBy || 'id',
      orderDir: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
    };

    const filterParams: Record<string, string | number | null> = {};
    Object.entries(tableManager.filters).forEach(([key, { value }]) => {
      if (key === 'id' && isValidFilterValue(value)) {
        filterParams[key] = Number(value);
      } else if (isValidFilterValue(value)) {
        filterParams[key] = value;
      }
    });

    if (isValidFilterValue(filterParams.keyword)) {
      filterParams.id = null;
      filterParams.name = null;
    } else if (isValidFilterValue(filterParams.id) || isValidFilterValue(filterParams.name)) {
      filterParams.keyword = null;
    }

    const newParams: VehicleTypePageFilterParams = {
      ...baseParams,
      ...filterParams
    };

    const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
    if (paramsChanged) {
      setParams(newParams);
    }
  }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order]);

  const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
    setParams(prev => ({
      ...prev,
      page: newPage
    }));
  };

  const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const size = parseInt(event.target.value, 10);
    if (!isNaN(size) && size > 0) {
      setParams(prev => ({
        ...prev,
        size,
        page: 0
      }));
    }
  };

  const userId = localStorage.getItem('userId');

  const handleStartExportVehicleTypes = () => {
    if (userId) {
      startExportFiltredVehicleTypes(userId, params);
    } else {
      enqueueSnackbar("Nous n'avons pas pu identifier l'utilisateur connecté.");
    }
  };

  return {
    ...tableManager,
    loading,
    error,
    totalVehicleTypes: totalElements || 0,
    onPageChange: handlePageChange,
    page: params.page,
    onRowsPerPageChange: handleRowsPerPageChange,
    rowsPerPage: params.size,
    sortBy: params.orderBy || 'id',
    sortDirection: params.orderDir || 'desc',
    filteredData: vehicleTypesPage,
    handleStartExportVehicleTypes: handleStartExportVehicleTypes,
    onResetFilters: onResetFilters
  };
}