"use client"

import React from "react"
import { TableCell, IconButton, Switch, Chip, Typography, Box } from "@mui/material"
import { TableColumn } from "@/shared/types/common"
import { TrajetStyledRow } from "@/shared/sections/contract/styles"
import FontAwesome from "@/shared/components/fontawesome"
import { faEdit } from "@fortawesome/free-solid-svg-icons"
import { VehicleType } from "@/shared/types/fleet-management/vehicleType"
import { useVehicleTypeStore } from "@/shared/api/stores/fleet-service/vehicle-type-store"
import TableActionColumn from "@/shared/components/table/table-action-column"

type Props = {
  row: VehicleType
  selected: boolean
  onSelectRow: VoidFunction
  handleEdit: (row: VehicleType) => void
  handleDetails: (row: VehicleType) => void
  columns: TableColumn[]
}

const TOOLTIP_TITLES = {
  edit: "Modifier type de véhicule",
  duplicate: "Copier type de véhicule",
  active: "Type de véhicule actif",
  inactive: "Type de véhicule inactif",
}

export default function VehicleTypeTableRow({
  row,
  selected,
  onSelectRow,
  handleEdit,
  handleDetails,
  columns,
}: Props) {

  const { toggleVehicleTypeActive } = useVehicleTypeStore();

  const handleToggleActivation = async () => {
    await toggleVehicleTypeActive(row.id, !row.active);
  };

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "id":
        return (
          <Typography variant="body2" sx={{ fontWeight: 500 }}>
            {value}
          </Typography>
        )

      case "name":
        return (
          <Typography variant="body2" sx={{ fontWeight: 500 }}>
            {value}
          </Typography>
        )

      case "active":
        return (
          <Chip
            label={value ? "Actif" : "Inactif"}
            size="small"
            color={value ? "success" : "default"}
            sx={{
              fontWeight: 500,
              minWidth: "70px"
            }}
          />
        )

      case "actions":
        return (
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <IconButton
              size="small"
              onClick={() => handleEdit(row)}
              sx={{
                color: "text.secondary",
                "&:hover": {
                  color: "primary.main",
                }
              }}
            >
              <FontAwesome icon={faEdit} />
            </IconButton>
            <Switch
              checked={row.active}
              onChange={() => handleToggleActivation()}
              size="small"
            />
          </Box>
        )

      default:
        return (
          <Typography variant="body2">
            {value}
          </Typography>
        )
    }
  }

  return (
    <TrajetStyledRow
      style={{ cursor: "pointer" }}
      sx={{
        "&:hover": {
          backgroundColor: "action.hover",
        }
      }}
      onDoubleClick={() => handleDetails(row)}
    >
      {columns.map((column: TableColumn) => (
        <TableCell
          key={column.id}
          align={column.align}
          sx={{
            py: 1.5
          }}
        >
          {renderCellContent(column.id, row[String(column.id) as keyof VehicleType])}
        </TableCell>
      ))}
      <TableCell align="right" sx={{ px: 1, whiteSpace: "nowrap" }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onToggle={(checked) => toggleVehicleTypeActive(row.id, checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </TrajetStyledRow>
  )
}