/**
 * Get the color for a given status
 * @param status - The status to get the color for
 * @returns The MUI color key for the status
 */
export const getStatusColor = (status: string): string => {
  switch(status) {
    case 'planned': return 'info.main';
    case 'in_progress': return 'warning.main';
    case 'completed': return 'success.main';
    default: return 'grey.500';
  }
};

/**
 * Get the French label for a given status
 * @param status - The status to get the label for
 * @returns The French label for the status
 */
export const getStatusLabel = (status: string): string => {
  switch(status) {
    case 'planned': return 'Planifié';
    case 'in_progress': return 'En cours';
    case 'completed': return 'Terminé';
    default: return status;
  }
};

/**
 * Get the French label for an intervention type
 * @param type - The type to get the label for
 * @returns The French label for the type
 */
export const getTypeLabel = (type: string): string => {
  return type === 'preventive' ? 'Entretien' : 'Maintenance';
};

/**
 * Format a duration in hours
 * @param duration - The duration in hours
 * @returns The formatted duration string
 */
export const formatDuration = (duration: number): string => {
  if (!duration && duration !== 0) return '0 heures';
  return `${duration} heures`;
};

/**
 * Format a date string or Date object into a Date object
 * @param date - The date to format
 * @returns The formatted Date object or null if invalid
 */
export const formatEventDate = (date: string | Date | null): Date | null => {
  if (!date) return null;
  try {
    const dateObj = new Date(date);
    return dateObj;
  } catch (e) {
    return null;
  }
};

/**
 * Validate and format time input
 * @param timeValue - The time string to validate
 * @returns The validated and formatted time string
 */
export const validateTimeInput = (timeValue: string): string => {
  // Remove any non-digit characters except colon
  let formattedTime = timeValue.replace(/[^\d:]/g, '');
  
  if (formattedTime.length <= 5) {
    // If there's a colon, validate hours and minutes
    if (formattedTime.includes(':')) {
      const [hours, minutes] = formattedTime.split(':');
      // Validate hours
      if (hours && parseInt(hours) > 23) {
        formattedTime = '23' + (minutes ? `:${minutes}` : ':');
      }
      // Validate minutes
      if (minutes && parseInt(minutes) > 59) {
        formattedTime = hours + ':59';
      }
    } else {
      // Add colon after 2 digits
      if (formattedTime.length >= 2) {
        formattedTime = formattedTime.slice(0, 2) + ':' + formattedTime.slice(2);
      }
    }
  }

  return formattedTime;
};

/**
 * Format a vehicle display string
 * @param vehicle - The vehicle object
 * @returns The formatted vehicle string
 */
export const formatVehicleDisplay = (vehicle: { 
  immatriculation: string; 
  marque: string; 
  modele: string; 
} | null | undefined): string => {
  if (!vehicle) return '';
  return `${vehicle.immatriculation} - ${vehicle.marque} ${vehicle.modele}`;
};

/**
 * Update time in a date object
 * @param date - The date to update
 * @param timeString - The time string in HH:mm format
 * @returns The updated date as ISO string or null if invalid
 */
export const updateDateWithTime = (date: Date | null, timeString: string): string | null => {
  if (!date || !timeString.match(/^\d{2}:\d{2}$/)) return null;

  const [hours, minutes] = timeString.split(':');
  const newDate = new Date(date);
  newDate.setHours(parseInt(hours));
  newDate.setMinutes(parseInt(minutes));
  
  return newDate.toISOString();
}; 